#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalBeaconDelay.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.DelayCalTarget
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalBeaconDelay(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("band", int, 3),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.384),
        Observation.ObsCalBase.scriptOption("subscanDuration", float, 30.),
        # ACA requires more than about 8.5 sec in FDM mode for 16-ant array
        # Observation.ObsCalBase.scriptOption("integrationDuration", float, 9.216),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 0.384),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "2 deg"),
        Observation.ObsCalBase.scriptOption("bbNames", str, ""),
        Observation.ObsCalBase.scriptOption("E", float, -140.79),
        Observation.ObsCalBase.scriptOption("N", float, -5189.69),
        Observation.ObsCalBase.scriptOption("U", float, 321.36 + 7. - 0.15),
        Observation.ObsCalBase.scriptOption("pointingOffsetFile", str, ""),
        Observation.ObsCalBase.scriptOption("delayOffsetFile", str, ""),
        Observation.ObsCalBase.scriptOption("focusOffsetFile", str, ""),
        # Observation.ObsCalBase.scriptOption("outFile", str, "pointingResults.txt"),
        Observation.ObsCalBase.scriptOption("frequency", float, -1.),
        Observation.ObsCalBase.scriptOption("bbFreqs", str, ""),
        Observation.ObsCalBase.scriptOption("refAnt", str, ""),
        Observation.ObsCalBase.scriptOption("corrMode", str, "TDM"),
        Observation.ObsCalBase.scriptOption("pol", str, "2"),
        Observation.ObsCalBase.scriptOption("autoOnly", bool, False),
        Observation.ObsCalBase.scriptOption("doATM", bool, False),
        Observation.ObsCalBase.scriptOption("useSQLDForATMProcessing", bool, False),
    ]

    def parseOptions(self):
        self.band                    = self.args.band
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.subscanDuration         = self.args.subscanDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        bbNameStr                    = self.args.bbNames
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)

        # self.doTsys = self.args.doTsys
        self.bPosE = self.args.E
        self.bPosN = self.args.N
        self.bPosU = self.args.U
        self.pointingOffsetFile = self.args.pointingOffsetFile
        self.delayOffsetFile = self.args.delayOffsetFile
        self.focusOffsetFile = self.args.focusOffsetFile
        # self.outFile = self.args.outFile
        self.frequency = self.args.frequency
        self.bbFreqs = self.args.bbFreqs
        self.refAnt = self.args.refAnt
        self.corrMode = self.args.corrMode
        self.pol = self.args.pol
        self.autoOnly = self.args.autoOnly

    def generateTunings(self):
        #
        # For B9/10 its useful to do an all-USB setup to avoid configing the
        # analysis into producing irrelevant LSB delays.
        #
        if self.band == 9:
            # can improve on this a lot by moving LOs around to minimise noise in the USB
            sameBBFreqs = True
            frequency = 668.0
            SBPref = "LSB"
        elif self.band == 10:
            sameBBFreqs = True
            frequency = 852.0
            SBPref = "LSB"
        else:
            sameBBFreqs = False
            frequency = Observation.SSRTuning.bandFreqs_delayMeasurement[self.band]
            SBPref = None

        if self.args.frequency > 0:
            frequency_ = self.args.frequency
            sameBBFreqs = True
        else:
            frequency_ = frequency

        if self.args.bbFreqs != "":
            bbFreqs = [float(token) for token in self.args.bbFreqs.split(",")]
        else:
            bbFreqs = None

        if self.corrMode == "FDM":
            bwd = 16
        else:
            bwd = None

        corrType = self._array.getCorrelatorType()
        self._spectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band=self.band,
                intent="interferometry_continuum",
                pol=self.pol,
                frequency=frequency_,
                bbNames=self.bbNames,
                SBPref=SBPref,
                sameBBFreqs=sameBBFreqs,
                bbFreqs=bbFreqs,
                corrType=corrType,
                corrMode=self.corrMode,
                bwd=bwd,
                dualMode=True,
                dump=self.dumpDuration,
                channelAverage=self.channelAverageDuration,
                integration=self.integrationDuration,
                autoOnly=self.autoOnly)

        if self.args.corrMode == "TDM":
            # Updated: 2019/10/15
            chanAverNumChans = 4
            if self.pol == '2':
                chanAverStartChan = (128 - chanAverStartChan) / 2
            elif self.pol == '4':
                chanAverStartChan = (64 - chanAverNumChans) / 2
            else:
                raise Exception("TODO")
            corrConfig = self._spectralSpec.BLCorrelatorConfiguration

            for bbc in corrConfig.BLBaseBandConfig:
                for spw in bbc.BLSpectralWindow:
                    chavg = spw.ChannelAverageRegion[0]
                    chavg.numberChannels = chanAverNumChans
                    chavg.startChannel = chanAverStartChan
        elif self.args.corrMode == "FDM":
            # FDM - will be removed anyway
            # chanAverNumChans = 512
            # chanAverStartChan = (3840 - 512) / 2
            # chanAverNumChans = 1024
            chanAverNumChans = 8
            spAvgFactor = 8
            # chanAverStartChan = 3840 - chanAverNumChans / 2
            factor = 1

            corrConfig = self._spectralSpec.BLCorrelatorConfiguration
            for bbc in corrConfig.BLBaseBandConfig:
                for spw in bbc.BLSpectralWindow:
                    spw.spectralAveragingFactor = spAvgFactor // factor
                    spw.effectiveNumberOfChannels //= factor

                    nCh = spw.effectiveNumberOfChannels
                    chavg = spw.ChannelAverageRegion[0]
                    chanAverStartChan = (nCh - chanAverNumChans) / 2

                    chavg.startChannel = chanAverStartChan // factor
                    chavg.numberChannels = chanAverNumChans // factor

    def setReferenceAntenna(self, refAnt=None):
        import Observation.Global
        if refAnt is None:
            refAnt = self.refAnt
        if Observation.Global.simulatedArray():
            return
        tcParameters = self.getTelCalParams()
        self.logInfo("Set %s as the reference antenna" % (refAnt))
        tcParameters.parameterTuning.setCalibParameterAsString('refantenna',
                                                               refAnt,
                                                               self.arrayName, 1)
    
    def doAtmCal(self):
        import CCL.SourceOffset
        import math
        if not self.args.doATM:
            return
        src = self._srcPointFocus
        ss = self._spectralSpec
        try:
            # ss = Observation.SSRTuning.generateAtmSpectralSpec(ss)
            atm = Observation.AtmCalTarget.AtmCalTarget(src, ss,
                                                        doHotLoad=True)
            atm.setOnlineProcessing(True)
            if self.args.useSQLDForATMProcessing:
                atm.setDataOrigin("TOTAL_POWER")
            else:
                atm.setDataOrigin('FULL_RESOLUTION_AUTO')
            atm.setDoZero(False)
            atm.setSubscanDuration(5.76)
            atm.setIntegrationTime(1.5)
            atm.setWVRCalReduction(True)
            atm.setApplyWVR(False)
            atm._referenceSource = atm._source
            atm._referenceOffset = CCL.SourceOffset.stroke(math.radians(600 / 3600.), 0, 0, 0, Control.HORIZON)
            self.logInfo('Executing AtmCal on ' + str(src.sourceName) + '...')
            atm.execute(self._obsmode)
            self.logInfo('Completed AtmCal on ' + str(src.sourceName))
        except BaseException as ex:
            import traceback
            self.logError(traceback.format_exc())
            msg = "Error executing AtmCal on source '%s'" % str(src.sourceName)
            self.logException(msg, ex)
            self.closeExecution(ex)
            raise ex

    def doDelayCal(self):
        src = self._srcPointFocus
        ss = self._spectralSpec
        try:
            delayCal = Observation.DelayCalTarget.DelayCalTarget(src, ss)
            delayCal.setSubscanDuration(self.subscanDuration)
            delayCal.setIntegrationTime(1.0)
            self.logInfo('Executing DelayCal on ' + src.sourceName + '...')
            delayCal.execute(self._obsmode)
            self.logInfo('Completed DelayCal on ' + src.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing cal survey scans on source %s" % src.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    # beacon test - override findPointFocusSource
    def findPointFocusSource(self):
        import numpy as np
        import CCL.FieldSource
        from Observation.ArtificialSourceHelper import ArtificialSourceHelper

        bPosENU = [self.bPosE, self.bPosN, self.bPosU]
        builder = ArtificialSourceHelper(bPosENU)
        builder.readOffsetsFromFile("pointing", self.pointingOffsetFile)
        builder.readOffsetsFromFile("delay", self.delayOffsetFile)
        # builder.readOffsetsFromFile("focus", self.focusOffsetFile)
        if self.refAnt == "":
            self.refAnt = builder.selectReferenceAntenna()
        src = builder.createFieldSource(referenceAntenna=self.refAnt)
        self.logInfo("beacon FieldSource: %s" % str(src.toxml()))
        self._srcPointFocus = src


obs = ObsCalBeaconDelay()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
# obs.setTelCalParams()
obs.logInfo("Executing DelayCal...")
obs.setReferenceAntenna()
obs.doDelayCal()
obs.doAtmCal()
obs.closeExecution()
