#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: StandardFastScan.py 247933 2017-08-08 15:57:43Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global math
import math

global Control
import Control

global CCL
import CCL.APDMSchedBlock

import traceback

global ControlExceptionsImpl
import ControlExceptionsImpl

global PyDataModelEnumeration
import PyDataModelEnumeration

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase
# import Observation.FillPattern
import Observation.FastScanPattern
import Observation.SchedulingBlock


class StandardFastScan(Observation.ObsCalBase.ObsCalBase):

    pointingExcursion = None
    mapAttenuatorSettingName = "Map attenuator setting"
    mapSubscanSequenceSpec = None
    mapMixerMode = Control.NORMAL
    _srcPointFocus = None
    # mapDuration = None

    def __init__(self):
        # 'SingleDish=False': Currently, we are using interferometric pointing
        # mode. As such, we cannot use the single dish observing mode.
        Observation.ObsCalBase.ObsCalBase.__init__(self, singleDish=False)
        sb = Observation.SchedulingBlock.SchedulingBlock()
        self._ssrSB = sb
        self.logInfo('Running %s scheduling block' % sb._controlSB.name)

    def parseOptions(self):

        target = self._ssrSB.getRepresentativeTarget()
        sp = target.getSpectralSpec()
        repBand =int(sp.FrequencySetup.receiverBand.replace("ALMA_RB_", ""))
        # Use some canned defaults for solar/moon maps in each band
        defMapSpacing = 20.0
        if repBand == 6:
            defMapSpacing = 10.0
        if repBand == 7:
            defMapSpacing = 8.0
        if repBand == 9:
            defMapSpacing = 4.0

        # !! These are on-source levels
        self.ifLevel = self.getExpertParameterOrDefault("ifLevel", float, -30.)
        self.bbLevel = self.getExpertParameterOrDefault("bbLevel", float, 2.)

        # TODO: We should be able to get rid of these parameters at some point.
        # map parameters
        self.mapPatternType          = self.getExpertParameterOrDefault("mapPatternType", str, "Area")
        self.mapPatternSubtype       = self.getExpertParameterOrDefault("mapPatternSubtype", str, "AUTO")
        self.mapWidth                = self.getExpertParameterOrDefault("mapWidth", float, 600.0)
        self.mapHeight               = self.getExpertParameterOrDefault("mapHeight", float, 600.0)
        self.mapSpacing              = self.getExpertParameterOrDefault("mapSpacing", float, defMapSpacing)
        self.mapOrientation          = self.getExpertParameterOrDefault("mapOrientation", float, 0.0)
        self.mapMaxFreq              = self.getExpertParameterOrDefault("mapMaxFreq", float, 1.0)
        # self.mapRepeats              = self.getExpertParameterOrDefault("mapRepeats", int, 2.0)

        # self.mapDoRef                = self.getExpertParameterOrDefault("mapDoRef", int, 1) > 0
        self.mapRefSubscanDuration   = self.getExpertParameterOrDefault("mapRefSubscanDuration", float, 10.0)
        # As we sometimes observe the Moon or Sun we are best to default to a long way away from the source!
        self.atmDoZero               = self.getExpertParameterOrDefault("atmDoZero", bool, True)
        self.elLimit                 = self.getExpertParameterOrDefault("ElLimit", str, "20 deg")
        self.activeSun               = self.getExpertParameterOrDefault("activeSun", bool, False)
        self.isSolar                 = self.getExpertParameterOrDefault("isSolar", bool, False)
        self.isRegression            = self.getExpertParameterOrDefault("isRegression", bool, False)
        if self.activeSun:
            self.mapMixerMode = Control.MD2

    def setTelCalParams(self):
        from Observation.Global  import simulatedArray
        self.logInfo("Setting TelCal parameters pointingFitWidth=False, simpleGaussianFit=False")
        if simulatedArray():
            return
        tcParameters = self.getTelCalParams()
        tcParameters.setCalibParameter('pointingFitWidth', False)
        tcParameters.setCalibParameter('simpleGaussianFit',False)

    def configureScanPatternUsingExpertParameters(self):
        import Observation.FillPattern
        self.logInfo("Try to generate map offset spec using expert parameters")
        frame = Control.HORIZON
        patternType = Observation.FillPattern.FillPatternType.getType(self.mapPatternType)
        patternSubType = Observation.FillPattern.FillPatternSubtype.getSubtype(self.mapPatternSubtype)
        fp = Observation.FillPattern.FillPattern('%.1f arcsec' % self.mapWidth,
                                                 '%.1f arcsec' % self.mapHeight,
                                                 '%.1f arcsec' % self.mapSpacing,
                                                 '%.3f deg' % self.mapOrientation,
                                                 patternType,
                                                 patternSubType)
        fsp = Observation.FastScanPattern.FastScanPattern(verbose=True)
        fsp.limitFrequencyMax = self.mapMaxFreq
        spec, duration = fsp.generatePatternSpec(fp, returnDuration=True)
        self.logInfo("Map pattern spec: " + str(spec._PatternSpec__idlSpec))
        mapOffsetSpec = Control.SourceOffset(longitudeOffset=0.0,
                                             latitudeOffset=0.0,
                                             longitudeVelocity=0.0,
                                             latitudeVelocity=0.0,
                                             pattern=spec._PatternSpec__idlSpec,
                                             frame=frame)
        self.logInfo("Map offset spec: " + str(mapOffsetSpec))
        for group in self._ssrSB.getGroupList():
            for target in group.getScienceTargetList():
                target.mapOffsetSpec = mapOffsetSpec
                target.singleMapDuration = duration

    def configureScanPatternUsingAPDMParameters(self):
        self.logInfo("Try to generate map offset spec using APDM parameters")
        for group in self._ssrSB.getGroupList():
            for target in group.getScienceTargetList():
                src = target.getSource()
                fp = src.getFieldPattern()
                if fp is None:
                    msg = "'%s' does not have FillPattern" % (target)
                    raise Exception(msg)
                self.logInfo("Field Patern: %s" % (fp.toDOM().toxml()))
                assert(fp is not None)
                verbose = True
                fsp = Observation.FastScanPattern.FastScanPattern(verbose=verbose)
                self.logInfo("Overridden limitFrequencyMax %.3f -> %.3f" % (fsp.limitFrequencyMax, self.mapMaxFreq))
                fsp.limitFrequencyMax = self.mapMaxFreq
                frame = fp.scanningCoordinateSystem
                self.logInfo("scanningCoordinateSystem = '%s'" % (fp.scanningCoordinateSystem))
                # FIXME: fp.scanningCoordinateSystem is not accepted by Control
                frame = Control.HORIZON
                assert(str(fp.scanningCoordinateSystem).lower() == str(frame).lower())
                lonOff = fp.patternCenterCoordinates.longitude.get()
                latOff = fp.patternCenterCoordinates.longitude.get()

                spec, duration = fsp.generatePatternSpec(fp, returnDuration=True)

                mapOffsetSpec = Control.SourceOffset(longitudeOffset=lonOff,
                                                     latitudeOffset=latOff,
                                                     longitudeVelocity=0.0,
                                                     latitudeVelocity=0.0,
                                                     pattern=spec._PatternSpec__idlSpec,
                                                     frame=frame)
                self.logInfo("Map offset spec: " + str(mapOffsetSpec))
                target.mapOffsetSpec = mapOffsetSpec
                target.singleMapDuration = duration

    def configureScanPattern(self):
        try:
            self.configureScanPatternUsingAPDMParameters()
        except:
            import traceback
            self.logWarning(traceback.format_exc())
            self.configureScanPatternUsingExpertParameters()


    def _updateScienceSourceForTests(self):
        import APDMEntities.SchedBlock
        groupList = self._ssrSB.getGroupList()
        for group in groupList[1:]:
            for target in group.getScienceTargetList():
                oldSrc = target.getSource()
                refPos = oldSrc.Reference[0]
                fp = oldSrc.getFieldPattern()
                import Observation.CalibratorCatalog
                cc = Observation.CalibratorCatalog.CalibratorCatalog('none')
                source, = cc.getBrightGridSources(minEl=25., maxEl=65.,
                                                  duration=2400.)
                src = source.buildFieldSource()
                src.Reference = [refPos]
                src.PointingPattern = None
                if isinstance(fp, APDMEntities.SchedBlock.FillPatternT):
                    src.FillPattern = fp
                target.setSource(src)
                self.logInfo("Regression: src=%s" % (src.toDOM().toxml()))

    def tweakATMTargets(self, target):
        import Observation.AtmCalTarget
        atmTargets = []
        for aTarget in target.getAssociatedCalTarget():
            if not isinstance(aTarget, Observation.AtmCalTarget.AtmCalTarget):
                continue
            atm = aTarget
            sp = atm.getSpectralSpec()
            assert(atm.getUseReferencePosition() == True)
            assert(atm.getOnlineProcessing())
            assert(atm.getApplyWVR() == False)
            atm.setDoZero(self.atmDoZero)
            atm.setMixerMode(self.mapMixerMode)
            if sp.hasCorrelatorConfiguration():
                atm.setDataOrigin('FULL_RESOLUTION_AUTO')
                atm.setWVRCalReduction(True)
            else:
                atm.setDataOrigin('TOTAL_POWER')
                atm.setWVRCalReduction(False)
            atm.setApplyWVR(False)
            atmTargets.append(atm)
        return atmTargets

    def observeTarget(self, target):
        import time
        from Observation.PointingCalTarget import PointingCalTarget
        from Observation.Global  import simulatedArray

        src = target.getSource()
        # Update some ATM target parameters
        atmTargets = self.tweakATMTargets(target)

        # Do pointing
        assocTargets = target.getAssociatedCalTarget()
        for assocTarget in assocTargets:
            if not isinstance(assocTarget, PointingCalTarget):
                continue
            if assocTarget.isNeeded(self._obsmode):
                assocTarget.execute(self._obsmode)

        # PRTSPR-35399
        mapSpectralSpec = target.getSpectralSpec()
        # self.logInfo("Calling setTarget to overrite 'prevSpectra'...")
        # self._obsmode.setTarget(src, mapSpectralSpec)
        # self.logInfo("setTarget - ended")
        self.logInfo("Calling setSpectrum to overrite 'prevSpectra'...")
        self._obsmode.setSpectrum(mapSpectralSpec)
        msg = "As setSpectrum does not wait for the completion of "
        msg += "tuning, wait for 23 seconds, just in case"
        self.logInfo(msg)
        if not simulatedArray():
            time.sleep(20.)

        # Do first ATM for the target
        for atmTarget in atmTargets:
            atmTarget.execute(self._obsmode)

        if len(atmTargets) == 0:
            cycleTime = 300.
        else:
            cycleTime = min([atmTarget.getCycleTime() for atmTarget in atmTargets])

        # Do Map
        if len(src.Reference) > 0:
            refSource, refOffset = target.parseReferencePosition(0)
        else:
            refOffset = None

        targetIntegrationTime = target.getIntegrationTime()
        mapDuration = target.singleMapDuration

        # How many maps we can make within the cycle time
        nMapCycle = int(float(cycleTime) / mapDuration)
        if (nMapCycle + 1) * mapDuration < cycleTime * 1.1:
            nMapCycle += 1

        while 1:
            cIntegTime = target.getCurrentIntegrationTime()
            remain = targetIntegrationTime - cIntegTime
            if remain <= 0:
                break

            # How many maps required to achieve the target integration time
            nMapRemain = int(float(remain) / mapDuration)
            if nMapRemain * mapDuration < remain:
                nMapRemain += 1

            nMap = min(nMapRemain, nMapCycle)
            if nMap <= 0:
                nMap = 1

            msg = "Integration time = [%.1f/%.1f]" % (cIntegTime, targetIntegrationTime)
            self.logInfo(msg)
            duration = mapDuration * nMap
            self.logInfo("Will do %d map(s) (=%.1f seconds)" % (nMap, duration))
            self.doMap(src, mapSpectralSpec, target.mapOffsetSpec, duration,
                       referenceOffset=refOffset)
            target.setCurrentIntegrationTime(cIntegTime + duration)
            # another ATM to bracket each target scan with ATM scans
            for atmTarget in atmTargets:
                atmTarget.execute(self._obsmode)

    def observeTargets(self):
        groupList = self._ssrSB.getGroupList()
        for group in groupList[1:]:
            msg = "[Group%s]" % (group.groupIndex)
            self.logInfo(msg)
            for target in group.getScienceTargetList():
                self.observeTarget(target)

    def prepareMapSubscanSequenceSpec(self, src, spectralSpec, mapOffsetSpec, mapDuration,
                                      referenceOffset=None):
        # Prepare subscan sequence specification
        ssl = CCL.SubscanList.SubscanList()
        subscanDuration = 0.048 * (1 + int(mapDuration / 0.048))
        ssl.addSubscan(src, spectralSpec, subscanDuration,
                       SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.ON_SOURCE,
                       CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                       PointingOffsetSpec    = mapOffsetSpec,
                       attenuatorSettingName = self.mapAttenuatorSettingName,
                       MixerMode             = self.mapMixerMode)
        if referenceOffset:
            # referenceOffset = CCL.SourceOffset.stroke(self.atmAzOffsetRad, 0, 0, 0, Control.HORIZON)
            self.logInfo("referenceOffset = %s" % str(referenceOffset))
            subscanDuration = 0.048 * (1 + int(self.mapRefSubscanDuration / 0.048))
            ssl.addSubscan(src, spectralSpec, subscanDuration,
                           SubscanIntent         = PyDataModelEnumeration.PySubscanIntent.OFF_SOURCE,
                           CalibrationDevice     = PyDataModelEnumeration.PyCalibrationDevice.NONE,
                           PointingOffsetSpec    = referenceOffset,
                           attenuatorSettingName = self.mapAttenuatorSettingName,
                           MixerMode             = self.mapMixerMode)
        subscanSpec = ssl.getSubscanSequenceSpecification()
        self.logInfo('subscanSpec.subscans = %s' % str(subscanSpec.subscans))
        self.mapSubscanSequenceSpec = subscanSpec

    def enableFocusModels(self, enable, continueOnError=False, noExcept=False):
        enableStr = "Enabling" if enable else "Disabling"
        exToRaise = None
        self.logInfo(enableStr + " focus models...")
        for ant in self._array.antennas():
            self.logInfo(enableStr + " focus model for antenna %s" % ant)
            try:
                 if not "OSS" in self.arrayName:
                     CCL.Mount.Mount(ant).enableFocusModel(enable)
            except BaseException as ex:
                print(ex)
                msg = "Error %s focus model for antenna %s" % (enableStr, ant)
                self.logError(msg)
                if continueOnError:
                    exToRaise = ex
                else:
                    raise ex
        if exToRaise is not None and not noExcept:
            raise exToRaise

    def executeMapScan(self, src):
        self.logInfo("Parking ACD so we optimise attenuators on-sky...")
        try:
            self._obsmode.setCalibrationDevice(PyDataModelEnumeration.PyCalibrationDevice.NONE)
        except BaseException as ex:
            self.logException('Exception thrown by obsmode.setCalibrationDevice(), considering this fatal!', ex)
            raise ex
        # Just before execution turn off the focus model when pointed at the source
        self.logInfo("Slewing to source with obsmode.setPointingDirection() to set focus before disabling focus models...")
        try:
            self._obsmode.setPointingDirection(src)
        except BaseException as ex:
            self.logException('Exception thrown by obsmode.setPointingDirection(), considering this fatal!', ex)
            raise ex
        try:
            self.enableFocusModels(enable=False, continueOnError=False, noExcept=False)
        except BaseException as ex:
            self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
            raise ex
        # the obvserving mode seems to set the levels after starting the motion pattern, so we have to do it ourselves and save the result. Ugh.
        self.logInfo("Optimising attenuators and saving to setting named '%s'" % self.mapAttenuatorSettingName)
        try:
            loMode = self._obsmode.getLOObservingMode()
            loMode.optimizeSignalLevels(ifTargetLevel=self.ifLevel,
                                        bbTargetLevel=self.bbLevel,
                                        settingName=self.mapAttenuatorSettingName)
        except BaseException as ex:
            self.logException('Exception thrown by optimizeSignalLevels() or getting reference to LOObservingMode, considering this fatal!', ex)
            raise ex
        # Sort out correlator calibration IDs
        if self.mapSubscanSequenceSpec.spectra[0].hasCorrelatorConfiguration():
            self.logInfo('Performing correlator calibration via getCalibrationIds() with validity interval %d seconds'
                                % self._obsmode.getCorrelatorCalibrationValidityInterval())
            try:
                self._obsmode.getCalibrationIds(self.mapSubscanSequenceSpec)
            except Exception as ex:
                self.logException('Exception thrown by obsmode.getCalibrationIds(), considering this fatal!', ex)
                raise ex
        else:
            self.logInfo('No correlator configuration, so just setting calibration IDs to -1 to save effort calling getCalibrationIds()')
            for s in self.mapSubscanSequenceSpec.subscans:
                s.calibrationId = -1
        # Execute the sequence
        self.logInfo("Now executing the subscan sequence...")
        try:
            self._obsmode.doSubscanSequence(self.mapSubscanSequenceSpec)
        except Exception as ex:
            self.logException('Exception thrown by obsmode.doSubscanSequence(), considering this fatal!', ex)
            raise ex
        self.logInfo('Subscan sequence finished')
        # Re-enable focus models, allowing an exception if any antennas fail
        try:
            self.enableFocusModels(enable=True, continueOnError=True, noExcept=False)
        except BaseException as ex:
            self.logException('Exception thrown by enableFocusModels(), considering this fatal!', ex)
            raise ex

    def doMap(self, src, spectralSpec, mapOffsetSpec, mapDuration, referenceOffset=None):
        self.prepareMapSubscanSequenceSpec(src, spectralSpec, mapOffsetSpec, mapDuration,
                                           referenceOffset=referenceOffset)
        scanIntent = CCL.ScanIntent.ScanIntent(PyDataModelEnumeration.PyScanIntent.OBSERVE_TARGET)
        # scanIntent = [CCL.ScanIntent.ScanIntent(PyDataModelEnumeration.PyScanIntent.OBSERVE_TARGET)]
        # scanIntent.append(CCL.ScanIntent.ScanIntent(PyDataModelEnumeration.PyScanIntent.CALIBRATE_POINTING, onlineProcessing=True, calibrationDataOrigin="TOTAL_POWER"))

        self.logInfo('Now beginning the scan')
        self._obsmode.beginScan(scanIntent)
        try:
            self.executeMapScan(src)
        except BaseException as ex:
            self.logError("Error executing map scan, cleaning up and exiting")
            self.enableFocusModels(enable=True, continueOnError=True, noExcept=True)
            self._obsmode.endScan()
            self.closeExecution(ex)
            raise ex
        self._obsmode.endScan()
        self.logInfo("Scan ended")

    def prepareTargets(self):
        from Observation.SSRTuning import overridePointingSpectralSpecs
        # To help testing, we override pointing band
        # FIXME: Provide obsmode or not
        overridePointingSpectralSpecs(self._ssrSB, self._array, self,
                                      obsmode=None)

        if self.isRegression:
            # For regression test, pick up a QSO which is up
            self._updateScienceSourceForTests()

        groupList = self._ssrSB.getGroupList()

        # Set up SQLDs
        for target in self._ssrSB.getTargetList():
            ss = target.getSpectralSpec()
            # TODO: 16ms for pointing targets, 1ms for science targets.
            if not hasattr(ss, 'SquareLawSetup') or ss.SquareLawSetup is None:
                msg = "Dual mode addition: adding SquareLawSetup to SpectralSpec named '%s'" % str(ss.name)
                self.logInfo(msg)
                ss.SquareLawSetup = CCL.APDMSchedBlock.SquareLawSetup()
                ss.SquareLawSetup.integrationDuration.set(0.001)

        # Prepare pointing targets
        for group in groupList:
            group.overwriteQueryCenterIfRequired(force=True)
            group.prepareTargets(assocPntToSciTarget=True)
            group.createAtmCalTargets(isSingleDish=True)
            group.printTargets("")

Observation.ObsCalBase.checkArray()

obs = StandardFastScan()
obs.parseOptions()
# obs.checkAntennas()
obs.startPrepareForExecution()
try:
    # Resolve pointing queries and generate ATM targets
    obs.prepareTargets()
    obs.configureScanPattern()
except BaseException as ex:
    obs.logError(traceback.format_exc())
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.setTelCalParams()
obs.logInfo("Executing AtmCal and Map...")
obs.observeTargets()
obs.closeExecution()
