#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalGrids.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global sys
import sys
global math
import math
global datetime
import datetime

global CCL
import CCL.Global
import CCL.FieldSource

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global AcsutilPy
import AcsutilPy.FindFile

global Observation
import Observation.AtmCalTarget
import Observation.DelayCalTarget
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.PhaseCalTarget
import Observation.BandpassCalTarget
import Observation.ObsCalBase
import Observation.ObsCalSources
import Observation.ShadowingChecker
import Observation.CalibratorCatalog


class ObsCalGrids(Observation.ObsCalBase.ObsCalBase):

    azoffset = math.radians(150.0/3600.0)

    options = [
        Observation.ObsCalBase.scriptOption("RepeatCount", int, 1),
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("HourAngleLimit", float, 3.0),
        Observation.ObsCalBase.scriptOption("NumTargets", int, 10),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 7),
        Observation.ObsCalBase.scriptOption("pipelineFriendly", bool, False),
        Observation.ObsCalBase.scriptOption("bandList", str, "7,3"),
        # Following two parameters are only relevant for pipeline mode.
        # Observation.ObsCalBase.scriptOption("allowHeterogeneousArray", bool, False),
        Observation.ObsCalBase.scriptOption("allowMultipleBands", bool, False),
        Observation.ObsCalBase.scriptOption("sourceListFile", str, ""),
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self._reverseSpecs = False
        self.doneBandpass = False
        self.donePrimaryAmpCal = False
        antennas = self._array.antennas()
        self._shadowingChecker = Observation.ShadowingChecker.ShadowingChecker(useSSRLogger=True,
                                                                               usedAntennaList=antennas)
        self.groupLSTBoundaries = [
            [1.75, 3.50],
            [5.50, 8.50],
            [10.00, 12.00],
            [14.75, 16.75],
            [17.75, 18.50],
            [19.75, 23.00],
        ]

        self.overlapSourceList = [
            ["J2258-2758", "J0635-7516"],
            ["J0423-0120", "J0854+2006"],
            ["J0635-7516", "J1256-0547"],
            ["J1229+0203", "J1642+3948"],
            ["J1550+0527", "J2025+3343", "J2148+0657"],
            ["J1924-2914", "J2357-5311"],
        ]

    def pipelineCheck(self):
        self.logInfo("Checking setup is compatible with PIPELINE requirements...")
        ants = self._array.antennas()
        errStr = ""
        num7m = 0
        num12m = 0
        if len(ants) < 2:
            errStr += " Array has less than two antennas."

        # ICT-13874: Cycle6 pipeline can handle 'mixed' array data sets
        # if N(7m) > N(12m)
        for a in ants:
            if a.startswith("CM"):
               num7m += 1
            else:
               num12m += 1
        self.logInfo("N(12m)=%d N(7m)=%d" % (num12m, num7m))
        if num7m > 0 and num12m > 0 and num12m >= num7m:
            errStr += " N(12m) is not less than N(7m) (see SCIREQ-1592)."

        if len(self.bandList) > 1 and not self.allowMultipleBands:
            errStr += " Observing band list contains more than one band (%s)." % str(self.bandList)
        if len(errStr) > 0:
            msg = "Pipeline-friendly mode requires all antnenas to be the same size, and only one observing band. Aborting due to:%s" % errStr
            ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            raise ex
        self.logInfo("Setup is compatible with PIPELINE requirements")

    def parseOptions(self):
        self.repeatCount             = self.getExpertParameterOrDefault("RepeatCount", int, 1)
        self.pointingSubscanDuration = self.getExpertParameterOrDefault("PointingSubscanDuration", float, 5.76)
        self.atmSubscanDuration      = self.getExpertParameterOrDefault("AtmSubscanDuration", float, 5.76)
        self.sbrSubscanDuration      = self.getExpertParameterOrDefault("SBRSubscanDuration", float, 5.76)
        self.focusSubscanDuration    = self.getExpertParameterOrDefault("FocusSubscanDuration", float, 5.76)
        self.dumpDuration            = self.getExpertParameterOrDefault("dumpDuration", float, 0.192)
        self.channelAverageDuration  = self.getExpertParameterOrDefault("channelAverageDuration", float, 0.576)
        self.integrationDuration     = self.getExpertParameterOrDefault("integrationDuration", float, 2.88)
        #self.atmIntegrationDuration  = self.getExpertParameterOrDefault("AtmIntegrationDuration", float, 0.576)
        self.tpIntegrationDuration   = self.getExpertParameterOrDefault("tpIntegrationDuration", float, 0.016)
        self.elLimit                 = self.getExpertParameterOrDefault("ElLimit", str, "20 deg")
        self.HourAngleLimit          = self.getExpertParameterOrDefault("HourAngleLimit", float, 3.0)
        self.numTargets              = self.getExpertParameterOrDefault("NumTargets", int, 10)
        self.pointFocusBand          = self.getExpertParameterOrDefault("pointFocusBand", int, 7)
        self.pipelineFriendly        = self.getExpertParameterOrDefault("pipelineFriendly", int, 0)
        bandStr                      = self.getExpertParameterOrDefault("bandList", str, "7,3")
        # Following two parameters are only relevant for pipeline mode.
        # self.allowHeterogeneousArray = self.getExpertParameterOrDefault("allowHeterogeneousArray", int, 0)
        self.allowMultipleBands      = self.getExpertParameterOrDefault("allowMultipleBands", int, 0)
        # ICT-17705
        self.sourceListFile          = self.args.sourceListFile
        self.bandList = []

        for s in bandStr.split(','):
            n = int(s)
            if n < 1 or n > 10:
                raise Exception("Invalid band number in band list: '%s'" % s)
            self.bandList.append(n)
        self.logInfo("Band list: %s" % str(self.bandList))
        if self.pipelineFriendly:
            self.pipelineCheck()

    def setTelCalParams(self):
        self.logInfo("Setting TelCal parameters ontheflyWVRcorrection=False spectrum=True")
        tcParameters = self.getTelCalParams()
        tcParameters.setCalibParameter('ontheflyWVRcorrection', False)
        # spectrum is to enable channel-by-channel bandpass result. Also binningFactor can be specified if desired.
        tcParameters.setCalibParameter('spectrum', True)
        # ICT-15218
        tcParameters.setCalibParameter("binningFactor", 1)

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        self._calSpectralSpecs = []
        for band in self.bandList:
            ss = self._tuningHelper.GenerateSpectralSpec(
                            band = band,
                            intent = "calsurvey",
                            corrType = corrType,
                            dualMode = True,
                            dump = self.dumpDuration,
                            channelAverage = self.channelAverageDuration,
                            integration = self.integrationDuration,
                            tpSampleTime = self.tpIntegrationDuration)
            ss.name = "Band %d calsurvey" % band
            self._calSpectralSpecs.append(ss)

    def pickupAvailableSSOsSingleBand(self, cc, spectralSpec,
                                      minEl=30,
                                      maxShadowFraction=0.5,
                                      maxResolvedFraction=0.5):
        # Calculate flux threshold
        minFluxList = cc.getRequiredCalibratorFlux("amplitude", el=40.,
                                                   spectralSpec=spectralSpec)
        minFlux = max(minFluxList)

        # Get available SSOs for this spectral spec
        wavelength = 299792458. / spectralSpec.getMeanFrequency()

        # First check the availability of SSOs with relaxed thresholds
        # (ICT-11561)
        chkSources = cc.getAvailableSSOs(wavelength,
                                         minEl=25.,
                                         maxEl=85.,
                                         duration=1200.,
                                         nRequest=-1,
                                         maxResolvedFraction=0.75,
                                         checkSourceNumbers=False,
                                         maxShadowFraction=0.5,
                                         fluxThreshold=minFlux / 3.,
                                         minimumBeamElongation=2. * 0.7,
                                         verbose=False)

        # Log
        chkSrcNameStr = ",".join([source.getName() for source in chkSources])
        values = (len(chkSources), chkSrcNameStr)
        msg = "%d source(s) passed the check selection criteria [%s]" % values
        self.logInfo(msg)

        # Next, check SSOs with more stringent criteria to pick up truly flux
        # calibrators.
        fluxSources = []
        for source in chkSources:
            fShadow = max(source._isObsData["fBlockedList"])
            minEl_ = min(source._isObsData["elList"])

            isIsolated = True
            for key, info in list(source._isObsData["separations"].items()):
                sep_ = info["separation"]
                minSep_ = (info["minimumSeparation"] - info["safeOccultLength"]) / 0.7
                if sep_ <= minSep_:
                    values = (source.getName(), key, sep_, minSep_)
                    self.logInfo("[%8s - %8s] sep=%5.1f minSep=%5.1f" % values)
                isIsolated &= (sep_ > minSep_)

            values = (source.getName(), source._isObsData["flux"], minEl_,
                      source._isObsData["fResolved"], fShadow, isIsolated)
            msgFmt = "[pickupAvailableSSOsSingleBand] %-8s flux=%4.1f [Jy]"
            msgFmt += " minEl=%5.1f [deg] fResolved=%5.2f fShadow=%5.2f"
            msgFmt += " isIsolated=%d"
            self.logInfo(msgFmt % values)

            if not isIsolated:
                continue
            if source._isObsData["flux"] < minFlux:
                continue
            if minEl_ < minEl:
                continue
            if source._isObsData["fResolved"] > maxResolvedFraction:
                continue
            if fShadow > maxShadowFraction:
                continue
            fluxSources.append(source)

        # Exclude flux sources from check sources
        chkSources = [s_ for s_ in chkSources if s_ not in fluxSources]

        # Log
        fluxSrcNameStr = ",".join([source.getName() for source in fluxSources])
        values = (len(fluxSources), fluxSrcNameStr)
        msg = "%d source(s) passed the fluxcal selection criteria [%s]" % values
        self.logInfo(msg)

        # msg = "Picked up %d sources (%s) and %d sources (%s)" % values
        # msg += " as primary and secondary calibrators, respectively,"
        # msg += " for spectral spec '%s'" % (spectralSpec)
        # self.logInfo(msg)
        return fluxSources, chkSources

    def pickupAvailableSSOs(self):
        """
        ICT-7107: pipeline friendly way of selecting SSO
        """
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing',
                                                             useSSRLogger=True)
        # ICT-9098: use alternative priority ordering of SSO list.
        cc.setGridSurveySSOPriorityOrder()
        sources = None

        fluxSources = []
        chkSources = []
        for iSS, ss in enumerate(self._calSpectralSpecs):
            fluxSources_, chkSources_= self.pickupAvailableSSOsSingleBand(cc, ss)

            for source in chkSources_:
                if source in chkSources:
                    continue
                chkSources.append(source)

            if iSS == 0:
                fluxSources.extend(fluxSources_)
            else:
                # Take intersection of sources
                fluxSources = [source for source in fluxSources if source in fluxSources_]

        for source in fluxSources:
            if source in chkSources:
                chkSources.remove(source)

        # ICT-15678: Ensure that there is only one target observed with
        # CALIBRATE_FLUX intent attached.
        for source in fluxSources[1:][::-1]:
            if source not in chkSources:
                chkSources.insert(0, source)
        if len(fluxSources) > 1:
            fluxSources = [fluxSources[0]]

        fluxSrcList = [source.buildFieldSource() for source in fluxSources]
        chkSrcList = [source.buildFieldSource() for source in chkSources]
        return fluxSrcList, chkSrcList

    def getLSTAndCurrentGroup(self):
        import numpy as np
        # Get current LST and identify group in which we are now.
        lst = self.sourceHelper.getLSTsecondOfDay() / 3600.
        groupMidLSTList = np.mean(self.groupLSTBoundaries, axis=1)
        diffLST = groupMidLSTList - lst
        diffLST[diffLST < -12.] += 24.
        diffLST[diffLST > 12.] -= 24.
        iGroup = np.argmin(np.abs(diffLST))
        return lst, iGroup + 1

    def pickupGridFluxCalibrator(self, grids):
        """
        ICT-8459: pick up an 'overlap' source, when no SSO is available
        as a primary flux calibrator.
        """
        import numpy as np
        # Identify the current group.
        lst, groupNumber = self.getLSTAndCurrentGroup()
        msg = "LST=%.2f [h]: Closest group=%d" % (lst, groupNumber)
        self.logInfo("[pickupGridFluxCalibrator] %s" % msg)

        iGroup = groupNumber - 1
        # Get FieldSource instance for overlap sources in the current group.
        candidates = []
        for sourceName in self.overlapSourceList[iGroup]:
            srcs = [g_ for g_ in grids if g_.sourceName == sourceName]
            if len(srcs) != 1:
                msg = "Cannot get source instance for '%s'" % sourceName
                self.logWarning(msg)
                continue
            candidates.append(srcs[0])
        if len(candidates) == 0:
            msg = "Cannot get any source instance for following sources"
            msg += " [%s]" % ", ".join(self.overlapSourceList[iGroup])
            ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            raise ex

        # Compare elevation of the overlap sources and take the highest one.
        elList = []
        for src in candidates:
            az, el = list(map(math.degrees, self.sourceHelper.sourceAzEl(src)))
            msg = "[%s] az=%6.1f el=%6.1f" % (src.sourceName, az, el)
            self.logInfo("[pickupGridFluxCalibrator]   %s" % msg)
            elList.append(el)
        iCandidate = np.argmax(elList)
        src = candidates[iCandidate]

        msg = "Picked up '%s'" % (src.sourceName)
        self.logInfo("[pickupGridFluxCalibrator] %s" % msg)
        return src

    def populateSourceList(self, nameList=None):
        # Estimate source flux for pointing/focus source for cal spectralspec
        # SPW center frequencies.
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing')
        cc.estimateSourceFlux(self._srcPointFocus, self._calSpectralSpecs,
                              setSourceProperties=True)
        self._calSources = [(self._srcPointFocus, "bandpass")]
        # For the convenience, we will populate source flux properties for
        # grid sources. To do this, we have to pass the list of spectral
        # setups to getAllGridSources() so that it knows for which frequencies
        # flux estimation has to be made. (Note that getAllGridSources() and
        # Observation.CalibratorSource.getFluxEstimate() were updated for this.
        grids = self.sourceHelper.getAllGridSources(self._calSpectralSpecs)

        # ICT-17705
        if self.sourceListFile != "":
            monitorQSOs, ampCalQSOs = self.readSourcesList(grids)
            grids = monitorQSOs

        # Pick up primary flux calibrator(s)
        if not self.pipelineFriendly:
            secondarySSOs = []
            ampCalSources = self.sourceHelper.getUnobscuredPrimaryFluxCalSources()
        else:
            primarySSOs, secondarySSOs = self.pickupAvailableSSOs()

            if len(primarySSOs) > 0:
                ampCalSources = list(primarySSOs)
                if self.sourceListFile != "":
                    grids.extend(ampCalQSOs)
            else:
                # No SSO available: will use one of overlap source as a
                # primary flux calibrator.
                if self.sourceListFile == "":
                    ampCalSources = [self.pickupGridFluxCalibrator(grids)]
                else:
                    ampCalSources = ampCalQSOs

        # Remove duplicate source entries
        grids = [src for src in grids if src.sourceName != self._srcPointFocus.sourceName]

        # Build the final list
        for src in ampCalSources:
            self._calSources.append((src, "amplitude"))
        for src in secondarySSOs:
            self._calSources.append((src, "check"))
        self.logInfo("Calibration source list:")
        for iSrc, (src, intent) in enumerate(self._calSources):
            self.logInfo("[%d] %s '%s'" % (iSrc, str(src.sourceName), intent))

        self._monitorSources = grids

    def readSourcesList(self, grids):
        import re
        pat_sep = re.compile("[ \t]+")

        gridNames = [src.sourceName for src in grids]

        lines = self._readFile(self.sourceListFile)

        monitorSrcs = []
        ampCalSrcs = []
        for iLine, line in enumerate(lines, 1):
            if len(line) == 0 or line.startswith("#"):
                continue
            if "#" in line:
                line = line[:line.index("#")]
            tokens = pat_sep.split(line)

            if len(tokens) == 2:
                srcName, intent = tokens
                if intent not in ["phase", "flux"]:
                    msg = "Unknown intent [%s] [L%d]" % (intent, iLine)
                    raise Exception(msg)
            elif len(tokens) == 1:
                srcName = tokens[0]
                intent = 'phase'
            else:
                values = (iLine, line)
                msg = "Unsupported format found in L%d (%s)" % values
                raise Exception(msg)

            if srcName in gridNames:
                if intent == 'phase':
                    monitorSrcs.append(srcName)
                else:
                    ampCalSrcs.append(srcName)
            else:
                msg = "Specified source (%s) is a grid source" % (srcName)
                raise Exception(msg)

        monitorSrcs = [src for src in grids if src.sourceName in monitorSrcs]
        ampCalSrcs = [src for src in grids if src.sourceName in ampCalSrcs]
        monitorSrcNames = ",".join([src.sourceName for src in monitorSrcs])
        ampCalSrcNames = ",".join([src.sourceName for src in ampCalSrcs])

        self.logInfo("QSOs to monitor: %s" % (monitorSrcNames))
        self.logInfo("QSO(a) as flux calibrator: %s" % (ampCalSrcNames))

        return monitorSrcs, ampCalSrcs

    def orderedSpecs(self):
        ret = self._calSpectralSpecs
        if self._reverseSpecs:
            ret = reversed(self._calSpectralSpecs)
        self._reverseSpecs = not self._reverseSpecs
        return ret

    def doPointing(self):
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            if not self.pipelineFriendly:
                pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def doFocus(self):
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(
                    SubscanFieldSource = self._srcPointFocus,
                    Axis = 'Z',
                    SpectralSpec = self._pointFocusSpectralSpec,
                    DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                    SubscanDuration = self.focusSubscanDuration,
                    OneWay = False,
                    NumPositions = 7)
            self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No focus results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def doATMCal(self, src, ss):
        atm = Observation.AtmCalTarget.AtmCalTarget(src, ss, doHotLoad=True)
        atm.setOnlineProcessing(True)
        atm.setDataOrigin('FULL_RESOLUTION_AUTO')
        atm.setDoZero(False)
        atm.setSubscanDuration(self.atmSubscanDuration)
        atm.setIntegrationTime(1.5)
        atm.setWVRCalReduction(True)
        # Applying the results takes a while with lots of
        # antennas, so until we use online WVR, don't bother
        atm.setApplyWVR(False)

        # Automatically adjust reference position for this target.
        atm.tweakReferenceOffset()
        self.logInfo('Executing AtmCal on ' + src.sourceName + '...')
        atm.execute(self._obsmode)
        self.logInfo('Completed AtmCal on ' + src.sourceName)

    def doCalSource(self, src, intent="phase"):
        from Observation.BandpassCalTarget import BandpassCalTarget
        from Observation.AmplitudeCalTarget import AmplitudeCalTarget
        from Observation.CheckSourceCalTarget import CheckSourceCalTarget
        from Observation.PhaseCalTarget import PhaseCalTarget

        az, el = Observation.ObsCalSources.ObsCalSources.sourceAzEl(src)
        fBlocked = self._shadowingChecker.isBlocked(math.degrees(az),
                                                    math.degrees(el),
                                                    verbose=True,
                                                    returnFraction=True)
        values = (src.sourceName, math.degrees(az), math.degrees(el), fBlocked)
        self.logInfo("[%s] (az, el)=(%.1f, %.1f) fBlocked=%.2f" % values)

        for ss in self.orderedSpecs():
            try:
                self.doATMCal(src, ss)

                if intent == 'bandpass':
                    ampliCal = BandpassCalTarget(src, ss)
                    ampliCal.setOnlineProcessing(True)
                elif intent == 'amplitude':
                    ampliCal = AmplitudeCalTarget(src, ss)
                elif intent == 'check':
                    # ICT-9098: assign check source intent to secondary SSOs.
                    ampliCal = CheckSourceCalTarget(src, ss)
                elif intent == 'phase':
                    ampliCal = PhaseCalTarget(src, ss)
                else:
                    assert(False)

                subscanDuration = 0.3e-9 * ss.getMeanFrequency()

                ampliCal.setSubscanDuration(subscanDuration)
                ampliCal.setIntegrationTime(1.0)
                self.logInfo('Executing AmplitudeCal on ' + src.sourceName + '...')
                ampliCal.execute(self._obsmode)
                self.logInfo('Completed AmplitudeCal on ' + src.sourceName)
            except BaseException as ex:
                import traceback
                self.logError(traceback.format_exc())
                msg = "Error executing cal survey scans on source %s" % src.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex

    def doCalObservations(self):
        from Observation.CalibratorSource import EquatorialCalibratorSource
        from Observation.CalSurveyScheduler import CalSurveyScheduler
        # Perform calibration scans
        for src, intent in self._calSources:
            self.doCalSource(src, intent)

        # Next observe grid sources
        sourceDict = dict()
        sources = []
        for src in self._monitorSources:
            ra = math.degrees(src.sourceCoordinates.longitude.get())
            dec = math.degrees(src.sourceCoordinates.latitude.get())
            source = EquatorialCalibratorSource(src.sourceName, ra, dec)
            sources.append(source)
            sourceDict[src.sourceName] = src

        scheduler = CalSurveyScheduler()
        # sources = sourceDict.keys()
        while 1:
            source = scheduler.selectNext(sources)
            if source is None:
                break
            src = sourceDict[source.getName()]
            self.doCalSource(src, "phase")
            sources.remove(source)
            if len(sources) == 0:
                break

    def _readFile(self, fPath):
        import os
        if not os.path.isfile(fPath):
            msg = "Unable to find the specified file '%s'" % (fPath)
            raise self.getIllegalParameterError(msg)
        with open(fPath) as f_:
            lines = [line.strip() for line in f_.readlines()]
        ret = []
        for line in lines:
            idx = line.find("#")
            if idx != -1:
                line = line[:idx]
            ret.append(line)
        return ret

survey = ObsCalGrids()
survey.parseOptions()
survey.checkAntennas()
survey.startPrepareForExecution()
try:
    survey.generateTunings()
    survey.findPointFocusSource()
    survey.populateSourceList()
    survey.setTelCalParams()
except BaseException as ex:
    import traceback
    print((traceback.format_exc()))
    survey.logException("Error in methods run during execution/obsmode startup", ex)
    survey.completePrepareForExecution()
    survey.closeExecution(ex)
    raise ex
survey.completePrepareForExecution()
survey.logInfo("Executing first pointing...")
survey.doPointing()
survey.logInfo("Executing second pointing -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing focus...")
survey.doFocus()
survey.logInfo("Executing third pointing after focus -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing Calibration observations...")
survey.doCalObservations()
survey.closeExecution()
