#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalFEDelays.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.DelayCalTarget
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalFEDelays(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("band", int, 4),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 1.536),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("bbNames", str, "")
    ]

    def parseOptions(self):
        self.band                    = self.args.band
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        bbNameStr                    = self.args.bbNames
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)

    def generateTunings(self):
        self._calSpectralSpecs = []
        corrType = self._array.getCorrelatorType()
        frequency = Observation.SSRTuning.bandFreqs_delayMeasurement[self.band]
        self._tuningHelper.band = self.band
        #
        # For 2 BBpr correlators we need two tunings to get both sidebands for 2SB RXs
        # TODO: if 2BBpr and one is 1/2 and the other is 3/4 we can proceed as for 3/4BBpr case
        #
        if self.bbNames is not None:
            self._tuningHelper.bbNames = self.bbNames
            bbprs = len(self.bbNames)
        else:
            bbprs = self._tuningHelper.corrSupportedBBprs()
        if bbprs > 2 or self.band > 8:
            # We just need one tuning
            if self.band == 9:
                # can improve on this a lot by moving LOs around to minimise noise in the USB
                sameBBFreqs = True
                frequency = 668.0
                SBPref = "LSB"
            elif self.band == 10:
                sameBBFreqs = True
                frequency = 852.0
                SBPref = "LSB"
            else:
                sameBBFreqs = False
                SBPref = None
            ss = self._tuningHelper.GenerateSpectralSpec(
                    intent = "interferometry_continuum",
                    frequency = frequency,
                    pol = '4',
                    SBPref = SBPref,
                    sameBBFreqs = sameBBFreqs,
                    corrType = corrType,
                    dualMode = True,
                    dump = self.dumpDuration,
                    channelAverage = self.channelAverageDuration,
                    integration = self.integrationDuration)
            ss.name = "Band %d delay" % self.band
            self._calSpectralSpecs.append(ss)
        else:
            # We need two tunings as 1/2BBpr 2SB.
            # This is a bit hacky - try to approximate the 4BBpr tuning.
            # Should really find a way to use the same LO1 to avoid retune time e.g. using hardware setups.
            IFcentre = 6.0e9
            if self.band == 6:
                IFcentre = 8.0e9
            ss = self._tuningHelper.GenerateSpectralSpec(
                    intent = "interferometry_continuum",
                    frequency = frequency - IFcentre,
                    pol = '4',
                    SBPref = "LSB",
                    corrType = corrType,
                    dualMode = True,
                    dump = self.dumpDuration,
                    channelAverage = self.channelAverageDuration,
                    integration = self.integrationDuration)
            ss.name = "Band %d LSB delay" % self.band
            self._calSpectralSpecs.append(ss)
            ss2 = self._tuningHelper.GenerateSpectralSpec(
                    intent = "interferometry_continuum",
                    frequency = frequency + IFcentre,
                    pol = '4',
                    SBPref = "USB",
                    corrType = corrType,
                    dualMode = True,
                    dump = self.dumpDuration,
                    channelAverage = self.channelAverageDuration,
                    integration = self.integrationDuration)
            ss2.name = "Band %d USB delay" % self.band
            self._calSpectralSpecs.append(ss2)
        for ss in self._calSpectralSpecs:
            self.logInfo("SpectralSpec '%s': %s" % (ss.name, ss.toxml()))


    def doDelayCals(self):
        src = self._srcPointFocus
        for ss in self._calSpectralSpecs:
            subscanDuration = 0.2e-9 * ss.getMeanFrequency()
            try:
                delayCal = Observation.DelayCalTarget.DelayCalTarget(src, ss)
                delayCal.setSubscanDuration(subscanDuration)
                delayCal.setIntegrationTime(1.0)
                self.logInfo('Executing DelayCal on ' + src.sourceName + '...')
                delayCal.execute(self._obsmode)
                self.logInfo('Completed DelayCal on ' + src.sourceName)
            except BaseException as ex:
                print(ex)
                msg = "Error executing cal survey scans on source %s" % src.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex




obs = ObsCalFEDelays()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing DelayCals...")
obs.doDelayCals()
obs.closeExecution()

