#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: FocusCalObservation.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase

class FocusCalObservation(Observation.ObsCalBase.ObsCalBase):

    specLine = False
    _srcPointFocus = None

    options = [
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("sourceName", str, ""),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 7),
        Observation.ObsCalBase.scriptOption("axisList", str, "Z"),
        Observation.ObsCalBase.scriptOption("numFocusPositions", int, 7),
        Observation.ObsCalBase.scriptOption("totalPower", bool, False),
        Observation.ObsCalBase.scriptOption("referenceBand", int, 7),
        Observation.ObsCalBase.scriptOption("use4x4mode", bool, False),
    ]

    def parseOptions(self):
        from PyDataModelEnumeration import PyPointingMethod
        from PyDataModelEnumeration import PyCalDataOrigin

        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.userSpecifiedSource     = self.args.sourceName
        self.pointFocusBand          = self.args.pointFocusBand
        self.numFocusPositions       = self.args.numFocusPositions
        self.totalPower              = self.args.totalPower
        self.referenceBand           = self.args.referenceBand
        self.use4x4mode              = self.args.use4x4mode
        axisStr                      = self.args.axisList
        self.axisList = []
        for s in axisStr.split(','):
            if s not in ["X", "Y", "Z"]:
                raise Exception("Invalid axis in axis list: '%s'" % s)
            self.axisList.append(s)
        self.logInfo("Axis list: %s" % str(self.axisList))

        if self.totalPower:
            self.dataOrigin = PyCalDataOrigin.TOTAL_POWER
            self.pointingMethod = PyPointingMethod.CROSS
            self.tuningIntent = "total_power"
            # Use single dish observing mode
            self.useSingleDishObservingMode = True

            # Following parameters are aligned with ObsCalSolarFastScan.py
            parameterList = [["PointingSubscanDuration", float, 3.072],
                             ["FocusSubscanDuration", float, 2.0],
                             ["numFocusPositions", int, 5]]
            for key, typ, defaultValue in parameterList:
                v_ = self.getExpertParameterOrDefault(key, typ, defaultValue)
                setattr(self.args, key, v_)
            self.pointingSubscanDuration = self.args.PointingSubscanDuration
            self.focusSubscanDuration = self.args.FocusSubscanDuration
            self.numFocusPositions = self.args.numFocusPositions
        else:
            self.dataOrigin = PyCalDataOrigin.CHANNEL_AVERAGE_CROSS
            self.pointingMethod = PyPointingMethod.FIVE_POINT
            self.tuningIntent = "interferometry_continuum"
        if not self.totalPower and self.pointFocusBand in [9,10]:
            self.specLine = True
        self.logInfo("totalPower=%s, pointFocusBand=%s, specLine=%s" % (str(self.totalPower), str(self.pointFocusBand), str(self.specLine)))

    def checkAntennas(self):
        if self.useSingleDishObservingMode:
            return
        return Observation.ObsCalBase.ObsCalBase.checkAntennas(self)

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        # Continuum default
        frequency = None
        bbFreqs = None
        numBB = None
        SBPref = None
        bwd = 1
        corrMode = None
        if self.specLine:
            numBB = 1
            bwd = 16
            corrMode = "FDM"
            if self.pointFocusBand == 9:
                # H2O maser line that is very bright in O-rich AGBs+RSGs
                bbFreqs = [658.005]
                SBPref = "LSB"
            elif self.pointFocusBand == 10:
                # HCN maser line that is very bright in Carbon stars
                bbFreqs = [890.7607]
                SBPref = "USB"
            else:
                msg = "Band %s currently not supported for spectral line tuning" % str(self.pointFocusBand)
                ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                ex.setData(Control.EX_USER_ERROR_MSG, msg)
                self.logError(msg)
                raise ex
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = self.tuningIntent,
                corrType = corrType,
                dualMode = True,
                bbFreqs = bbFreqs,
                SBPref = SBPref,
                bwd = bwd,
                corrMode = corrMode,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        if self.specLine:
            self._pointFocusSpectralSpec.name += " specLine"
            self._pointFocusSpectralSpec.FrequencySetup.dopplerReference = "rest"
            spAverFactor = 4
            # 512 chans at (1875/16)MHz BW is about 7km/s, similar to maser line widths
            chanAverNumChans = 512
            chanAverStartChan = (3840 - 512) / 2
            if self._pointFocusSpectralSpec.hasBLCorrelatorConfiguration():
                from PyDataModelEnumeration import PyCorrelationBit
                if self.use4x4mode:
                    # Use 4x4 correlation, which means dividing all the channel numbers by 4.
                    factor = 4
                    correlationBits = PyCorrelationBit.BITS_4x4
                else:
                    factor = 1
                    correlationBits = PyCorrelationBit.BITS_2x2
                for bbc in self._pointFocusSpectralSpec.BLCorrelatorConfiguration.BLBaseBandConfig:
                    for spw in bbc.BLSpectralWindow:
                        spw.correlationBits = correlationBits
                        spw.effectiveNumberOfChannels //= factor
                        spw.spectralAveragingFactor = spAverFactor // factor
                        spw.ChannelAverageRegion[0].numberChannels = chanAverNumChans // factor
                        spw.ChannelAverageRegion[0].startChannel = chanAverStartChan // factor
            elif self._pointFocusSpectralSpec.hasACACorrelatorConfiguration():
                for bbc in self._pointFocusSpectralSpec.ACACorrelatorConfiguration.ACABaseBandConfig:
                    for spw in bbc.ACASpectralWindow:
                        spw.spectralAveragingFactor = spAverFactor
                        spw.ChannelAverageRegion[0].numberChannels = chanAverNumChans
                        spw.ChannelAverageRegion[0].startChannel = chanAverStartChan

        if self.isReferenceBandPointingRequired():
            corrType = self._array.getCorrelatorType()
            sp = self._tuningHelper.GenerateSpectralSpec(
                band=self.referenceBand,
                intent=self.tuningIntent,
                corrType=corrType,
                dualMode=True,
                bwd=1,
                dump=self.dumpDuration,
                channelAverage=self.channelAverageDuration,
                integration=self.integrationDuration,
                tpSampleTime=self.tpIntegrationDuration
            )
            sp.name = "Band %d pointing/focus" % (self.referenceBand)
            self._referencePointFocusSpectralSpec = sp

    def isReferenceBandPointingRequired(self):
        if self.pointFocusBand in [9, 10] and not self.specLine:
            return True
        return False

    def doPointing(self, useReferenceBand=False):
        if useReferenceBand:
            sp = self._referencePointFocusSpectralSpec
        else:
            sp = self._pointFocusSpectralSpec
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, sp)
            pointingCal.setPointingMethod(self.pointingMethod)
            pointingCal.setDataOrigin(self.dataOrigin)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDelayCalReduction(not self.totalPower)
            pointingCal.setWVRCalReduction(not self.totalPower)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            pointingCal.checkAndApplyResult(self._obsmode)
        except BaseException as ex:
            import traceback
            self.logError("%s" % (traceback.format_exc()))
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
                    
    def doFocus(self):
        import CCL.APDMSchedBlock
        if self.totalPower:
            if len(self._srcPointFocus.Reference) == 0:
                ref = CCL.APDMSchedBlock.Reference()
                ref.setCoordinates('-600 arcsec', 0, u'horizon')
                ref.cycleTime.set(1.0)
                ref.subScanDuration.set(self.focusSubscanDuration)
                ref.integrationTime.set(self.focusSubscanDuration)
                self._srcPointFocus.Reference = [ref]

        for axis in self.axisList:
            try:
                focusCal = Observation.FocusCalTarget.FocusCalTarget(
                        SubscanFieldSource = self._srcPointFocus,
                        Axis = axis,
                        SpectralSpec = self._pointFocusSpectralSpec,
                        DataOrigin = self.dataOrigin,
                        SubscanDuration = self.focusSubscanDuration,
                        OneWay = False,
                        NumPositions = self.numFocusPositions)
                focusCal.setWVRCalReduction(not self.totalPower)
                # if self.totalPower:
                #     focusCal.setIntegrationTime(3.0 * self.focusSubscanDuration)

                self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
                focusCal.execute(self._obsmode)
                self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
                result = focusCal.checkResult(self._array)
                self.logInfo("Result is: %s" % str(result))
                if len(result) > 0: 
                    for key in list(result.keys()):
                        self.logInfo("Found solution for %s using polarization(s) %s" %
                                (key, result[key]))
                    focusCal.applyResult(self._obsmode, result)
                else:
                    if not "OSS" in self._array._arrayName:
                        raise Exception("No focus results!")
            except BaseException as ex:
                import traceback
                self.logError("%s" % (traceback.format_exc()))
                msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex

    def dopplerCorrect(self):
        """
        Doppler correction for epctral line case
        """
        if not self.specLine:
            return
        self.logDebug("SpectralSpec before Doppler: %s" % str(self._pointFocusSpectralSpec.toxml()))
        corrected = self._obsmode.dopplerShift(self._pointFocusSpectralSpec, self._srcPointFocus, 0)
        self.logDebug("SpectralSpec after Doppler: %s" % str(corrected.toxml()))
        self._pointFocusSpectralSpec = corrected


obs = FocusCalObservation()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource(minEl=30.0,
                             useThisSource=obs.userSpecifiedSource,
                             singleDish=obs.totalPower,
                             specLine=obs.specLine,
                             band=obs.pointFocusBand)
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
# Doppler needs obsmode, so after completePrepareForExecution
obs.dopplerCorrect()
if obs.isReferenceBandPointingRequired():
    # ICT-9912: Do pointing scans in the reference band for band9/10 TP focus SBs
    obs.logInfo("Executing first reference band pointing...")
    obs.doPointing(useReferenceBand=True)
    obs.logInfo("Executing second reference band pointing...")
    obs.doPointing(useReferenceBand=True)
obs.logInfo("Executing first pointing...")
obs.doPointing()
obs.logInfo("Executing second pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing focus...")
obs.doFocus()
obs.logInfo("Executing last pointing -- make sure results are good!...")
obs.doPointing()
obs.closeExecution()
