#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalWeakSourceSurvey.py 241594 2017-02-27 21:53:18Z javarias $"

#
# forcing global imports is due to an OSS problem
#
global CCL
import CCL.Global
import CCL.FieldSource

global Control
import Control
global ControlExceptionsImpl
import ControlExceptionsImpl

global AcsutilPy
import AcsutilPy.FindFile

global Observation
import Observation.ObsCalBase
import Observation.AtmCalTarget
import Observation.DelayCalTarget
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ScanList
import Observation.SSRTuning

global math
import math

global os
import os


class ObsCalWeakSourceSurvey(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("RepeatCount", int, 2),
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        #Observation.ObsCalBase.scriptOption("AtmIntegrationDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("maxSources", int, 30),
        Observation.ObsCalBase.scriptOption("sourcesPerCal", int, 4),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 7),
        Observation.ObsCalBase.scriptOption("logFile", str, "/groups/science/data/WeakSourceSurveyLog.txt"),
        Observation.ObsCalBase.scriptOption("useSpecialBandSwitching", int, 1),
        Observation.ObsCalBase.scriptOption("bandList", str, "3"),
        Observation.ObsCalBase.scriptOption("sourceListFile", str, ""),
        Observation.ObsCalBase.scriptOption("subscanDuration", float, -1),
        Observation.ObsCalBase.scriptOption("ampSubscanDuration", float, -1)
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self._reverseSpecs = False
        self.WVRResult = None
        self.gridList = []
        self.sourceList = []
        self.observedSources = []
        self.doneBandpass = False
        self.coneRadius = 10.0

    def parseOptions(self):
        self.repeatCount             = self.args.RepeatCount
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.atmSubscanDuration      = self.args.AtmSubscanDuration
        self.sbrSubscanDuration      = self.args.SBRSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        #self.atmIntegrationDuration  = self.args.AtmIntegrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.maxSources              = self.args.maxSources
        self.sourcesPerCal           = self.args.sourcesPerCal
        self.pointFocusBand          = self.args.pointFocusBand
        self.logFile                 = self.args.logFile
        self.useSpecialBandSwitching = self.args.useSpecialBandSwitching
        bandStr                      = self.args.bandList
        self.sourceListFile          = self.args.sourceListFile
        self.subscanDuration         = self.args.subscanDuration
        self.ampSubscanDuration      = self.args.ampSubscanDuration

        self.bandList = []
        for s in bandStr.split(','):
            n = int(s)
            if n < 1 or n > 10:
                msg = "Invalid band number in band list: '%s'" % s
                ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                ex.setData(Control.EX_USER_ERROR_MSG, msg)
                self.logError(msg)
                raise ex
            self.bandList.append(n)
        self.logInfo("Band list: %s" % str(self.bandList))

    def setTelCalParams(self):
        self.logInfo("Setting TelCal parameters ontheflyWVRcorrection=%s spectrum=True" % (str(self.haveOperationalWVR)))
        tcParameters = self.getTelCalParams()
        tcParameters.setCalibParameter('ontheflyWVRcorrection', self.haveOperationalWVR)
        # spectrum is to enable channel-by-channel bandpass result. Also binningFactor can be specified if desired.
        tcParameters.setCalibParameter('spectrum', True)
        # ICT-15218
        tcParameters.setCalibParameter("binningFactor", 1)

    def generateTuningsSpecialB3B7(self):
        corrType = self._array.getCorrelatorType()

        # band 3 we'll FLOOG high, band 7 we'll FLOOG low :)

        fPhot = 95.03e9
        fFLOOG = 32.5e6
        fLO1_B3 = fPhot + fFLOOG
        fLO1_B7 = 3.0 * (fPhot - fFLOOG)
        fLO2 = [9.979e9, 8.021e9, 8.021e9, 9.979e9]
        fSky_B3 = [fLO1_B3-fLO2[0]+3.0e9, fLO1_B3-fLO2[1]+3.0e9, fLO1_B3+fLO2[2]-3.0e9, fLO1_B3+fLO2[3]-3.0e9]
        fSky_B7 = [fLO1_B7-fLO2[0]+3.0e9, fLO1_B7-fLO2[1]+3.0e9, fLO1_B7+fLO2[2]-3.0e9, fLO1_B7+fLO2[3]-3.0e9]
        useUSB = [False, False, True, True]
        #
        # A bit ugly: make a normal SS, then turn it into a hardware one
        #
        spec3 = self._tuningHelper.GenerateSpectralSpec(
                band = 3,
                intent = "calsurvey",
                frequency = fLO1_B3,
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration)
        spec3.FrequencySetup.lO1Frequency.set(fLO1_B3)
        spec3.FrequencySetup.isUserSpecifiedLO1 = True
        spec3.FrequencySetup.hasHardwareSetup = True
        spec3.FrequencySetup.floog.set(fFLOOG)
        spec3.FrequencySetup.tuneHigh = True
        spec3.FrequencySetup.dopplerReference = u'topo'
        for i in range(len(spec3.FrequencySetup.BaseBandSpecification)):
            bbs = spec3.FrequencySetup.BaseBandSpecification[i]
            bbs.lO2Frequency.set(fLO2[i])
            bbs.centerFrequency.set(fSky_B3[i])
            bbs.useUSB = useUSB[i]
            bbs.use12GHzFilter = False
        spec3.name = "Band 3 calsurvey special hardware setup"

        spec7 = self._tuningHelper.GenerateSpectralSpec(
                band = 7,
                intent = "calsurvey",
                frequency = fLO1_B7,
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration)
        spec7.FrequencySetup.lO1Frequency.set(fLO1_B7)
        spec7.FrequencySetup.isUserSpecifiedLO1 = True
        spec7.FrequencySetup.hasHardwareSetup = True
        spec7.FrequencySetup.floog.set(fFLOOG)
        spec7.FrequencySetup.tuneHigh = False
        spec7.FrequencySetup.dopplerReference = u'topo'
        for i in range(len(spec7.FrequencySetup.BaseBandSpecification)):
            bbs = spec7.FrequencySetup.BaseBandSpecification[i]
            bbs.lO2Frequency.set(fLO2[i])
            bbs.centerFrequency.set(fSky_B7[i])
            bbs.useUSB = useUSB[i]
            bbs.use12GHzFilter = False
        spec7.name = "Band 7 calsurvey special hardware setup"

        self._calSpectralSpecs = [spec7, spec3]

        for ss in self._calSpectralSpecs:
            Observation.SSRTuning.setSpWSBsFromSignalPathSBs(ss, self)

        self.logInfo(spec3.toxml())
        self.logInfo(spec7.toxml())
        

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        self._calSpectralSpecs = []

        # Should check we are to use B3,7
        if self.useSpecialBandSwitching:
            self.generateTuningsSpecialB3B7()
        else:
            for band in self.bandList:
                ss = self._tuningHelper.GenerateSpectralSpec(
                            band = band,
                            intent = "calsurvey",
                            corrType = corrType,
                            dualMode = True,
                            dump = self.dumpDuration,
                            channelAverage = self.channelAverageDuration,
                            integration = self.integrationDuration,
                            tpSampleTime = self.tpIntegrationDuration)
                ss.name = "Band %d calsurvey" % band
                self._calSpectralSpecs.append(ss)

    def readSourceList(self, fname="config/WeakSourceList.csv"):
        if self.sourceListFile == "":
            filePath = AcsutilPy.FindFile.findFile(fname)[0]
        else:
            filePath = os.path.expanduser(self.sourceListFile)
        self.logInfo("Reading source list from file '%s'..." % filePath)
        if filePath == '':
            msg = "Unable to find source list file '%s'" % fname
            ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
        fp = open(filePath, 'r')
        self._schedule = []
        for line in fp:
            line = line.strip()
            if len(line)==0 or not line[0].isdigit():
                continue
            subs = line.split(',', 32)
            if len(subs) < 32:
                self.logWarning("Skipping invalid line: '%s'" % line)
                continue
            name = subs[1]
            ra = float(subs[29])
            decl = float(subs[30])
            grid = subs[31].split('|',1)[0]
            #print name, ra, decl, grid
            if grid not in self.gridList:
                self.gridList.append(grid)
            self.sourceList.append({"name": name, "ra": ra, "decl": decl, "grid": grid})
        #self.logInfo("Full source list: %s" % str(self.sourceList))
        self.logInfo("Loaded %d sources" % (len(self.sourceList)))
        self.logInfo("Full grid list: %s" % str(self.gridList))

    def rankGridList(self):
        gridSources = self.sourceHelper.getSources(self.gridList)
        lstSec = self.sourceHelper.getLSTsecondOfDay()
        for src in gridSources:
            ra   = math.degrees(src.sourceCoordinates.longitude.get())
            decl = math.degrees(src.sourceCoordinates.latitude.get())
            raSec = int(ra * (3600.0 / 15.0))
            secToTransit = raSec - lstSec
            if secToTransit >  12*3600: secToTransit -= 24*3600
            if secToTransit < -12*3600: secToTransit += 24*3600
            transitZenithDist = abs(decl - self.sourceHelper.siteLatitude)
            metric = abs(secToTransit - 3600)
            # CloudSat / zenith avoidance - coneRadius needs to be a compromise to avoid overly biassing to low elevation
            if(transitZenithDist < self.coneRadius):
                minSecToTransit = int((transitZenithDist + self.coneRadius) * 3600.0/15.0) + 2.0*3600.0
                minSecAfterTransit = int((transitZenithDist + self.coneRadius) * 3600.0/15.0)
                self.logInfo("%s: minSecToTransit=%d, minSecAfterTransit=%d" % (str(src.sourceName), minSecToTransit, minSecAfterTransit))
                if secToTransit < minSecToTransit and -secToTransit < minSecAfterTransit:
                    self.logInfo("%s: down-weighting due to zenith avoidance" % (str(src.sourceName)))
                    metric = 100000 + (minSecAfterTransit + secToTransit)
            self.logInfo("%s: secToTransit=%d, transitZenithDist=%.3f deg, metric=%d" % (str(src.sourceName), secToTransit, transitZenithDist, metric))
            src.metric = metric
        gridSources.sort(key=lambda k: k.metric)
        for s in gridSources:
            self.logInfo("%s: metric = %d" % (str(s.sourceName), s.metric))
        self.gridSourcesRanked = gridSources

    def readObservingLog(self):
        if self.logFile.lower() == 'NoLog'.lower():
            self.logInfo("readObservingLog(): log file is '%s', assuming no sources observed already" % self.logFile)
            return
        try:
            fp = open(os.path.expanduser(self.logFile), 'r')
        except:
            self.logInfo("readObservingLog(): log file '%s' not readable, assuming no sources observed already" % self.logFile)
            return
        numLines = 0
        for line in fp:
            if len(line)==0 or line[0] == '\n' or line[0] == '#':
                continue
            numLines += 1
            name = line.split(',',2)[0]
            if name not in self.observedSources:
                self.observedSources.append(name)
        self.logInfo("readObservingLog(): read %d source lines, now have %d sources observed already" % (numLines, len(self.observedSources)))

    def pickBestGrid(self):
        self._srcPointFocus = None
        numToObserve = 0
        for grid in self.gridSourcesRanked:
            nSources = 0
            for s in self.sourceList:
                if s['grid'] != str(grid.sourceName):
                    continue
                if s['name'] in self.observedSources:
                    continue
                nSources += 1
            import Observation.CalibratorSource
            ra, dec = list(map(math.degrees, grid.getRADec()))
            source = Observation.CalibratorSource.EquatorialCalibratorSource(str(grid.sourceName), ra, dec,
                                                                             useSSRLogger=True)
            isObs = source.isObservable(minEl=20., duration=2400., verbose=True)
            if not isObs:
                self.logInfo("pickBestGrid(): Skipping grid %s as it is not observable" % str(grid.sourceName))
                continue
            if nSources > 0:
                self._srcPointFocus = grid
                numToObserve = nSources
                break
            self.logInfo("pickBestGrid(): Skipping grid %s as no unobserved sources for it" % str(grid.sourceName))
        if self._srcPointFocus is None:
            msg = "pickBestGrid(): No unobserved sources found! Maybe try again in a couple of hours."
            ex = ControlExceptionsImpl.TransientExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, msg)
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
        self.logInfo("pickBestGrid(): Picked grid source '%s' with %d sources to observe" % (str(self._srcPointFocus.sourceName), numToObserve))
        if numToObserve > self.maxSources:
            factor = int(1 + (numToObserve // self.maxSources))
            self.maxSources = 1 + (numToObserve // factor)
            self.logInfo("pickBestGrid(): Will observe first %d sources" % self.maxSources)

    def populateSourceList(self):
        self._calSources = []
        n = 0
        for s in self.sourceList:
            if s['grid'] != str(self._srcPointFocus.sourceName):
                continue
            if s['name'] in self.observedSources:
                continue
            src = CCL.FieldSource.EquatorialSource(sourceName = s['name'], ra = math.radians(s['ra']), dec = math.radians(s['decl']))
            self._calSources.append(src)
            self.logInfo(src.toxml())
            n += 1
            if n >= self.maxSources:
                break

    def orderedSpecs(self):
        ret = self._calSpectralSpecs
        if self._reverseSpecs:
            ret = list(reversed(self._calSpectralSpecs))
        self._reverseSpecs = not self._reverseSpecs
        return ret

    def doPointing(self):
        try:
            # Currently just a test
            try:
                self.isObservable(self._srcPointFocus, 600)
            except BaseException as ex:
                self.logException('Exception thrown by isObservable() when checking source %s, considering this fatal!' % self._srcPointFocus.sourceName, ex)
                self.closeExecution(ex)
                raise ex
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            # The source is selected by the user and thus it may not be strong
            # enough to allow Delay reduction.
            # pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + str(self._srcPointFocus.sourceName) + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + str(self._srcPointFocus.sourceName))
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    msg = "No pointing results!"
                    ex = ControlExceptionsImpl.FatalExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    self.closeExecution(ex)
                    raise ex
        except BaseException as ex:
            msg = "Error executing pointing on source '%s'" % str(self._srcPointFocus.sourceName)
            self.logException(msg, ex)
            self.closeExecution(ex)
            raise ex
                    
    def doFocus(self):
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(
                    SubscanFieldSource = self._srcPointFocus,
                    Axis = 'Z',
                    SpectralSpec = self._pointFocusSpectralSpec,
                    DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                    SubscanDuration = self.focusSubscanDuration,
                    OneWay = False,
                    NumPositions = 7)
            self.logInfo('Executing FocusCal on ' + str(self._srcPointFocus.sourceName) + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + str(self._srcPointFocus.sourceName))
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    msg = "No focus results!"
                    ex = ControlExceptionsImpl.FatalExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    self.closeExecution(ex)
                    raise ex
        except BaseException as ex:
            msg = "Error executing focus on source '%s'" % str(self._srcPointFocus.sourceName)
            self.logException(msg, ex)
            self.closeExecution(ex)
            raise ex
                    
    def doAtmCals(self):
        doWVR = True
        for ss in self.orderedSpecs():
            try:
                atm = Observation.AtmCalTarget.AtmCalTarget(self._srcPointFocus, ss, doHotLoad=True)
                atm.setOnlineProcessing(True)
                atm.setDataOrigin('FULL_RESOLUTION_AUTO')
                atm.setDoZero(False)
                atm.setSubscanDuration(self.atmSubscanDuration)
                atm.setIntegrationTime(1.5)
                atm.setWVRCalReduction(self.haveOperationalWVR)
                atm.setApplyWVR(False)
                self.logInfo('Executing AtmCal on ' + str(self._srcPointFocus.sourceName) + '...')
                atm.execute(self._obsmode)
                self.logInfo('Completed AtmCal on ' + str(self._srcPointFocus.sourceName))
                if not self.haveOperationalWVR:
                    return
                # For now keep this as a sanity check, but we no longer apply the result
                if doWVR and ss.hasBLCorrelatorConfiguration():
                    self.WVRResult = atm.checkWVRResult(self._array)
                    self.logInfo("Retrieved WVR result: %s" % str(self.WVRResult))
                    if not "OSS" in self._array._arrayName:
                        if self.WVRResult is None:
                            msg = "doAtmCals(): WVR Result is None, aborting execution"
                            ex = ControlExceptionsImpl.FatalExImpl()
                            ex.setData(Control.EX_USER_ERROR_MSG, msg)
                            self.logError(msg)
                            self.closeExecution(ex)
                            raise ex
                    doWVR = False
            except BaseException as ex:
                msg = "Error executing AtmCal on source '%s'" % str(self._srcPointFocus.sourceName)
                self.logException(msg, ex)
                self.closeExecution(ex)
                raise ex

    def doAmpCal(self):
        calSrc = self._srcPointFocus
        scanList = Observation.ScanList.ScanList()
        for ss in self.orderedSpecs():
            try:
                if self.ampSubscanDuration < 0:
                    # hack to get about 20s B3, 40s B7
                    subscanDuration = math.pow(1.0e-9*ss.getMeanFrequency(), 0.65)
                else:
                    subscanDuration = self.ampSubscanDuration
                isObs = False
                try:
                    isObs = self.isObservable(calSrc, subscanDuration+10)
                except BaseException as ex:
                    self.logException('Exception thrown by isObservable() when checking source %s, considering this fatal!' % calSrc.sourceName, ex)
                    self.closeExecution(ex)
                    raise ex
                if not isObs:
                    msg = "Calibration source '%s' is not observable!!" % str(calSrc.sourceName)
                    ex = ControlExceptionsImpl.FatalExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    self.closeExecution(ex)
                    raise ex
                ampCal = Observation.AmplitudeCalTarget.AmplitudeCalTarget(calSrc, ss)
                if not self.doneBandpass:
                    ampCal._enableBandpassCalReduction = True
                ampCal.setSubscanDuration(subscanDuration)
                ampCal.setIntegrationTime(1.0)
                self.logInfo('Executing AmpCal on ' + str(calSrc.sourceName) + '...')
                ampCal.execute(self._obsmode, scanList)
                #self.logInfo('Completed AmpCal on ' + str(calSrc.sourceName))
                self.logInfo('Prepared AmpCal on ' + str(calSrc.sourceName))
            except BaseException as ex:
                msg = "Error executing amplitude/phase calibration scan on source '%s'" % str(calSrc.sourceName)
                self.logException(msg, ex)
                self.closeExecution(ex)
                raise ex
        try:
            scanList.execute(self._obsmode)
        except BaseException as ex:
            # scanList.execute already logs the exception
            self.closeExecution(ex)
            raise ex
        self.doneBandpass = True

    def doCalSource(self, src):
        scanList = Observation.ScanList.ScanList()
        for ss in self.orderedSpecs():
            if self.subscanDuration < 0:
                # hack to get about 20s B3, 40s B7
                subscanDuration = math.pow(1.0e-9*ss.getMeanFrequency(), 0.65)
            else:
                subscanDuration = self.subscanDuration
            try:
                phaseCal = Observation.PhaseCalTarget.PhaseCalTarget(src, ss)
                phaseCal.setSubscanDuration(subscanDuration)
                phaseCal.setIntegrationTime(1.0)
                self.logInfo('Executing PhaseCal on ' + str(src.sourceName) + '...')
                phaseCal.execute(self._obsmode, scanList)
                self.logInfo('Prepared PhaseCal on ' + str(src.sourceName))
            except BaseException as ex:
                msg = "Error preparing cal survey scans on source '%s'" % str(src.sourceName)
                self.logException(msg, ex)
                self.closeExecution(ex)
                raise ex
        try:
            scanList.execute(self._obsmode)
        except BaseException as ex:
            # scanList.execute already logs the exception
            self.closeExecution(ex)
            raise ex

    def writeLogEntry(self, sourceName):
        if self.logFile.lower() == 'NoLog'.lower():
            self.logInfo("writeLogEntry(): log file is '%s', won't write log entry" % self.logFile)
            return
        oldUmask = os.umask(0)
        fd = os.open(os.path.expanduser(self.logFile), os.O_WRONLY | os.O_CREAT | os.O_APPEND, 0o666)
        fp = os.fdopen(fd, "a")
        fp.write("%s, %s\n" % (str(sourceName), str(self.uid)))
        fp.close()
        os.umask(oldUmask)

    def doCalObservations(self):
        sourceScan = 0
        calSrc = self._srcPointFocus
        for i in range(self.repeatCount):
            self.doPointing()
            self.logInfo("Executing AtmCal...")
            self.doAtmCals()
            for src in self._calSources:
                if sourceScan % self.sourcesPerCal == 0:
                    self.doAmpCal()
                isObs = False
                try:
                    isObs = self.isObservable(src, 60)
                except BaseException as ex:
                    self.logException('Exception thrown by isObservable() when checking source %s, considering this fatal!' % src.sourceName, ex)
                    self.closeExecution(ex)
                    raise ex
                if not isObs:
                    self.logInfo("Skipping source '%s' as not observable" % str(src.sourceName))
                    continue
                self.doCalSource(src)
                sourceScan += 1
                self.writeLogEntry(src.sourceName)
        isObs = False
        try:
            isObs = self.isObservable(calSrc, 60)
        except BaseException as ex:
            self.logException('Exception thrown by isObservable() when checking source %s, considering this fatal!' % calSrc.sourceName, ex)
            self.closeExecution(ex)
            raise ex
        if isObs:
            self.doAmpCal()
            

obs = ObsCalWeakSourceSurvey()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.readSourceList()
    obs.rankGridList()
    obs.readObservingLog()
    obs.pickBestGrid()
    obs.populateSourceList()
    obs.checkForOperationalWVR()
    obs.setTelCalParams()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing first pointing...")
obs.doPointing()
obs.logInfo("Executing second pointing -- make sure results are good!...")
obs.doPointing()
obs.logInfo("Executing focus...")
obs.doFocus()
obs.logInfo("Executing Calibration observations...")
obs.doCalObservations()
obs.closeExecution()

