#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalSBRSweep.py 214235 2015-02-06 05:01:13Z nphillip $"

#
# forcing global imports is due to an OSS problem
#
global sys
import sys
global math
import math
global datetime
import datetime

global CCL
import CCL.Global
import CCL.FieldSource

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global AcsutilPy
import AcsutilPy.FindFile

global Observation
import Observation.AtmCalTarget
import Observation.DelayCalTarget
import Observation.PointingCalTarget
import Observation.FocusCalTarget
# import Observation.TargetUtility
import Observation.SchedulingBlock
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalSBRSweep(Observation.ObsCalBase.ObsCalBase):

    """
    From Band.java in LOsolutions package, which is used to validate BB freqs:
      //       receiverband           band               IF   warm cold lodriver
      //                            low   high     SB  low hi  mul  mul low    hi
    Band1( ReceiverBand.ALMA_RB_01, 31.3, 45.0, SB.USB, 4, 12,  1,   1, 27.3, 33.0),
    Band2( ReceiverBand.ALMA_RB_02, 67.0, 90.0, SB.LSB, 4, 12,  6,   1, 79.0, 94.0),
    Band3( ReceiverBand.ALMA_RB_03, 84.0,116.0, SB.SB2, 4,  8,  6,   1, 92.0,108.0),
    Band4( ReceiverBand.ALMA_RB_04,125.0,163.0, SB.SB2, 4,  8,  3,   2, 66.5, 77.5),
    Band5( ReceiverBand.ALMA_RB_05,163.0,211.0, SB.SB2, 4,  8,  2,   6, 28.5, 34.0),
    Band6( ReceiverBand.ALMA_RB_06,211.0,275.0, SB.SB2, 5, 10,  6,   3, 73.6, 88.4),
    Band7( ReceiverBand.ALMA_RB_07,275.0,373.0, SB.SB2, 4,  8,  6,   3, 94.3,121.7),
    Band8( ReceiverBand.ALMA_RB_08,385.0,500.0, SB.SB2, 4,  8,  3,   6, 65.5, 82.0),
    Band9( ReceiverBand.ALMA_RB_09,602.0,720.0, SB.DSB, 4, 12,  3,   9, 67.8, 79.1),
    Band10(ReceiverBand.ALMA_RB_10,787.0,950.0, SB.DSB, 4, 12,  6,   9, 83.3,104.7);
    """
    bandInfo = {
        1:  {'skyMin':  31.3e9, 'skyMax':  45.0e9, 'IFMin': 4.0e9, 'IFMax': 12.0e9, 'multWarm': 1.0, 'multCold': 1.0},
        2:  {'skyMin':  67.0e9, 'skyMax':  90.0e9, 'IFMin': 4.0e9, 'IFMax': 12.0e9, 'multWarm': 6.0, 'multCold': 1.0},
        3:  {'skyMin':  84.0e9, 'skyMax': 116.0e9, 'IFMin': 4.0e9, 'IFMax':  8.0e9, 'multWarm': 6.0, 'multCold': 1.0},
        4:  {'skyMin': 125.0e9, 'skyMax': 163.0e9, 'IFMin': 4.0e9, 'IFMax':  8.0e9, 'multWarm': 3.0, 'multCold': 2.0},
        5:  {'skyMin': 163.0e9, 'skyMax': 211.0e9, 'IFMin': 4.0e9, 'IFMax':  8.0e9, 'multWarm': 2.0, 'multCold': 6.0},
        6:  {'skyMin': 211.0e9, 'skyMax': 275.0e9, 'IFMin': 5.0e9, 'IFMax': 10.0e9, 'multWarm': 6.0, 'multCold': 3.0},
        7:  {'skyMin': 275.0e9, 'skyMax': 373.0e9, 'IFMin': 4.0e9, 'IFMax':  8.0e9, 'multWarm': 6.0, 'multCold': 3.0},
        8:  {'skyMin': 385.0e9, 'skyMax': 500.0e9, 'IFMin': 4.0e9, 'IFMax':  8.0e9, 'multWarm': 3.0, 'multCold': 6.0},
        9:  {'skyMin': 602.0e9, 'skyMax': 720.0e9, 'IFMin': 4.0e9, 'IFMax': 12.0e9, 'multWarm': 3.0, 'multCold': 9.0},
        10: {'skyMin': 787.0e9, 'skyMax': 950.0e9, 'IFMin': 4.0e9, 'IFMax': 12.0e9, 'multWarm': 6.0, 'multCold': 9.0}
    }

    options = [
        Observation.ObsCalBase.scriptOption("RepeatCount", int, 1),
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        #Observation.ObsCalBase.scriptOption("AtmIntegrationDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("NumTargets", int, 10),
        Observation.ObsCalBase.scriptOption("pointFocusBand", int, 8),
        Observation.ObsCalBase.scriptOption("bandList", str, "8")
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self._doSBRatio = True
        self._reverseSpecs = False

    def checkAntennas(self):
        if len(self._array.antennas()) < 2:
            raise Exception("Array has less than two antennas, this won't work!")
        if len(self._array.antennas()) < 3:
            self.logInfo("Array contains less than three antennas, so won't waste time with SBRatio measurements!")
            self._doSBRatio = False

    def parseOptions(self):
        self.repeatCount             = self.args.RepeatCount
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.atmSubscanDuration      = self.args.AtmSubscanDuration
        self.sbrSubscanDuration      = self.args.SBRSubscanDuration
        self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        #self.atmIntegrationDuration  = self.args.AtmIntegrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.numTargets              = self.args.NumTargets
        self.pointFocusBand          = self.args.pointFocusBand
        bandStr                      = self.args.bandList
        self.bandList = []
        for s in bandStr.split(','):
            n = int(s)
            if n < 1 or n > 10:
                raise Exception("Invalid band number in band list: '%s'" % s)
            self.bandList.append(n)
        self.logInfo("Band list: %s" % str(self.bandList))

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()
        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.pointFocusBand,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.pointFocusBand
        self._calSpectralSpecs = []
        for band in self.bandList:
            f0 = self.bandInfo[band]['skyMin'] + self.bandInfo[band]['IFMax']
            f1 = self.bandInfo[band]['skyMax'] - self.bandInfo[band]['IFMax']
            step = self.bandInfo[band]['IFMax'] - self.bandInfo[band]['IFMin']
            nStep = int(math.ceil((f1 - f0) / step))
            step = (f1 - f0) / nStep
            nTunings = nStep + 1
            self.logInfo("Will sweep from %.3f to %.3f GHz with %d tunings stepped %.3f GHz apart" % (1.0e-9*f0, 1.0e-9*f1, nTunings, 1.0e-9*step))
            for i in range(nTunings):
                f = f0 + i*step
                self.logInfo("Generating tuning for frequency %.3f GHz..." % (1.0e-9*f))
                ss = self._tuningHelper.GenerateSpectralSpec(
                            band = band,
                            frequency = f,
                            intent = "interferometry_continuum",
                            LO_offset_strategy = 2,
                            corrType = corrType,
                            dualMode = True,
                            dump = self.dumpDuration,
                            channelAverage = self.channelAverageDuration,
                            integration = self.integrationDuration,
                            tpSampleTime = self.tpIntegrationDuration)
                ss.name = "Band %d centre %.3f GHz SBR" % (band, 1.0e-9*f)
                self._calSpectralSpecs.append(ss)

    def orderedSpecs(self):
        ret = self._calSpectralSpecs
        if self._reverseSpecs:
            ret = reversed(self._calSpectralSpecs)
        self._reverseSpecs = not self._reverseSpecs
        return ret

    def doPointing(self):
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
                    
    def doFocus(self):
        try:
            focusCal = Observation.FocusCalTarget.FocusCalTarget(
                    SubscanFieldSource = self._srcPointFocus,
                    Axis = 'Z',
                    SpectralSpec = self._pointFocusSpectralSpec,
                    DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                    SubscanDuration = self.focusSubscanDuration,
                    OneWay = False,
                    NumPositions = 7)
            self.logInfo('Executing FocusCal on ' + self._srcPointFocus.sourceName + '...')
            focusCal.execute(self._obsmode)
            self.logInfo('Completed FocusCal on ' + self._srcPointFocus.sourceName)
            result = focusCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0: 
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                focusCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No focus results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing focus on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex
                    
    def doSBRatios(self):
        if not self._doSBRatio:
            return
        for ss in self.orderedSpecs():
            try:
                subscanDuration = 0.1e-9 * ss.getMeanFrequency()
                sbrCal = Observation.SBRatioCalTarget.SBRatioCalTarget(
                    SubscanFieldSource = self._srcPointFocus,
                    SpectralSpec = ss,
                    DataOrigin = 'CHANNEL_AVERAGE_CROSS',
                    SubscanDuration = subscanDuration,
                    IntegrationTime = subscanDuration)
                self.logInfo('Executing SBRatioCal on ' + self._srcPointFocus.sourceName + '...' )
                sbrCal.execute(self._obsmode)
                self.logInfo('Completed SBRatioCal on ' + self._srcPointFocus.sourceName)
            except BaseException as ex:
                print(ex)
                msg = "Error executing SBRatio on source %s" % self._srcPointFocus.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex
            # Doing this in here is a bit ugly, but convenient.
            # Idea is to get a real-time display of SBRs from TelCalSpy,
            # plus to measure BB detector zero points
            #try:
            #    atm = Observation.AtmCalTarget.AtmCalTarget(
            #        SubscanFieldSource = self._srcPointFocus,
            #        SpectralSpec = ss,
            #        DataOrigin = 'FULL_RESOLUTION_AUTO',
            #        doZero = False,
            #        SubscanDuration = self.atmSubscanDuration,
            #        IntegrationTime = 1.5,
            #        doHotLoad = True)
            #    atm.setOnlineProcessing(True)
            #    #try:
            #    #    sqlTest = ss.SquareLawSetup.integrationDuration
            #    #    atm.setDataOrigin('TOTAL_POWER')
            #    #    atm.setDoZero(True)
            #    #except: pass
            #    atm.setWVRCalReduction(True)
            #    atm.setApplyWVR(True)
            #    self.logInfo('Executing AtmCal on ' + self._srcPointFocus.sourceName + '...')
            #    atm.execute(self._obsmode)
            #    self.logInfo('Completed AtmCal on ' + self._srcPointFocus.sourceName)
            #except BaseException, ex:
            #    print ex
            #    msg = "Error executing AtmCal on source %s" % self._srcPointFocus.sourceName
            #    self.logError(msg)
            #    self.closeExecution(ex)
            #    raise ex



survey = ObsCalSBRSweep()
survey.parseOptions()
survey.checkAntennas()
survey.startPrepareForExecution()
survey.generateTunings()
survey.findPointFocusSource()
survey.completePrepareForExecution()
survey.logInfo("Executing first pointing...")
survey.doPointing()
survey.logInfo("Executing second pointing -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing focus...")
survey.doFocus()
survey.logInfo("Executing third pointing after focus -- make sure results are good!...")
survey.doPointing()
survey.logInfo("Executing SBRatios...")
survey.doSBRatios()
#survey.logInfo("Executing Calibration observations...")
#survey.doCalObservations()
survey.closeExecution()
