#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalAzSearch.py 239891 2017-01-16 20:20:03Z nphillip $"

#
# forcing global imports is due to an OSS problem
#
global math
import math
global copy
import copy

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global PyDataModelEnumeration
import PyDataModelEnumeration

global CCL
import CCL.SourceOffset
import CCL.FrontEnd
import CCL.LPR
import CCL.IFSwitch
import CCL.WCA1
import CCL.WCA2
import CCL.WCA3
import CCL.WCA4
import CCL.WCA5
import CCL.WCA6
import CCL.WCA7
import CCL.WCA8
import CCL.WCA9
import CCL.WCA10

global Observation
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.ObsCalBase

class ObsCalBandCheckout(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("bandList", str, "3"),
        Observation.ObsCalBase.scriptOption("checkAllBands", bool, False),
        Observation.ObsCalBase.scriptOption("atmSubscanDuration", float, 4.0),
        Observation.ObsCalBase.scriptOption("dummySubscanDuration", float, 10.0),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.008),
        Observation.ObsCalBase.scriptOption("excursion", float, 1000.0),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "2 deg"),
        #Observation.ObsCalBase.scriptOption("band", int, 3),
        Observation.ObsCalBase.scriptOption("atmAzOffsetArcsec", float, 300.0)
    ]

    bandFreqs = {
         1:  41.0e9, # maximum supported while production LSs are not updatred
         2:  89.5e9, # avoid WVR 91.6GHz birdie in both 4BB and 2BB cases for AtmCal
         3: 101.0e9, # avoid WVR LO at 91.6GHz
         4: 150.0e9,
         5: 183.0e9,
         6: 230.0e9,
         7: 345.0e9,
         8: 410.0e9,
         9: 669.0e9,
        10: 861.0e9
    }

    # We consider the LO power was off if LOPA Vd is less than this value
    LOPAVdThreshold = 0.2 # Volts

    ants = []
    fes = []
    lprs = []
    ifss = []
    # wcas is two dimensional, band being the outer axis
    wcas = []

    bandsToCheck = []

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self, singleDish=True)
        self._srcPointFocus = None


    def parseOptions(self):
        bandStr                      = self.args.bandList
        self.checkAllBands           = self.args.checkAllBands
        self.dummySubscanDuration    = self.args.dummySubscanDuration
        self.atmSubscanDuration      = self.args.atmSubscanDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.excursion               = self.args.excursion
        self.elLimit                 = self.args.ElLimit
        self.atmAzOffsetArcsec       = self.args.atmAzOffsetArcsec
        self.atmAzOffsetRad = math.radians(self.atmAzOffsetArcsec / 3600.0)
        self.bandList = []
        for s in bandStr.split(','):
            n = int(s)
            if n < 1 or n > 10:
                msg = "Invalid band number in band list: '%s'" % s
                ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                ex.setData(Control.EX_USER_ERROR_MSG, msg)
                self.logError(msg)
                raise ex
            self.bandList.append(n)
        self.logInfo("Band list: %s" % str(self.bandList))


    def generateTunings(self):
        self.dummyScanSpectralSpecs = []
        self.atmCalSpectralSpecs = []
        for band in self.bandsToCheck:
            # Use two SpectralSpecs a bit offset to force a re-tuning between scans
            ds = self._tuningHelper.GenerateSpectralSpec(
                    band = band,
                    frequency = self.bandFreqs[band] - 1.0e9,
                    intent = "total_power",
                    tpSampleTime = self.tpIntegrationDuration)
            ds.name = "Band %d dummy scan" % band
            self.dummyScanSpectralSpecs.append(ds)
            ts = self._tuningHelper.GenerateSpectralSpec(
                    band = band,
                    frequency = self.bandFreqs[band],
                    intent = "total_power",
                    tpSampleTime = self.tpIntegrationDuration)
            ts.name = "Band %d AtmCal" % band
            self.atmCalSpectralSpecs.append(ts)


    def getCommonAvailableBands(self):
        availablePerAnt = self._array.getAvailableBands()
        available = copy.copy(availablePerAnt[0].bands)
        for i in range(1, len(availablePerAnt)):
            toRemove = []
            for b in available:
                if b not in availablePerAnt[i].bands:
                    toRemove.append(b)
            for b in toRemove:
                available.remove(b)
        return available
   
    def getCommonPoweredBands(self):
        poweredPerAnt = self._array.getPoweredBands()
        powered = copy.copy(poweredPerAnt[0].bands)
        for i in range(1, len(poweredPerAnt)):
            toRemove = []
            for b in powered:
                if b not in poweredPerAnt[i].bands:
                    toRemove.append(b)
            for b in toRemove:
                powered.remove(b)
        return powered

    def getPoweredBandStats(self):
        poweredPerAnt = self._array.getPoweredBands()
        allPowered = copy.copy(poweredPerAnt[0].bands)
        for i in range(1, len(poweredPerAnt)):
            for b in poweredPerAnt[i].bands:
                if b not in allPowered:
                    allPowered.append(b)
        numAnts = [0]*len(allPowered)
        for i in range(len(allPowered)):
            b = allPowered[i]
            for pa in poweredPerAnt:
                if b in pa.bands:
                    numAnts[i] += 1
        ret = {}
        for i in range(len(allPowered)):
            ret[allPowered[i]] = numAnts[i]
        return ret

    def getPowerOffBands(self, bandsToUse):
        numPowered = self.getPoweredBandStats()
        poweredPerAnt = self._array.getPoweredBands()
        toPowerOff = []
        for pa in poweredPerAnt:
            bands = copy.copy(pa.bands)
            for b in toPowerOff:
                if b in pa.bands:
                    bands.remove(b)
            for b in bandsToUse:
                if b in pa.bands:
                    bands.remove(b)
            while(True):
                if len(bands) <= (3 - len(bandsToUse)):
                    break
                # find the least popular one and turn it off
                minNum = 9999; minBand = None
                for i in range(len(bands)):
                    if numPowered[bands[i]] < minNum:
                        minNum = numPowered[bands[i]]
                        minBand = bands[i]
                toPowerOff.append(minBand)
                bands.remove(minBand)
        return toPowerOff


    def getWCAConstructors(self):
        c = []
        for b in self.bandList:
            if b == 1:
                c.append(CCL.WCA1.WCA1)
            elif b ==2:
                c.append(CCL.WCA2.WCA2)
            elif b ==3:
                c.append(CCL.WCA3.WCA3)
            elif b ==4:
                c.append(CCL.WCA4.WCA4)
            elif b ==5:
                c.append(CCL.WCA5.WCA5)
            elif b ==6:
                c.append(CCL.WCA6.WCA6)
            elif b ==7:
                c.append(CCL.WCA7.WCA7)
            elif b ==8:
                c.append(CCL.WCA8.WCA8)
            elif b ==9:
                c.append(CCL.WCA9.WCA9)
            elif b ==10:
                c.append(CCL.WCA10.WCA10)
            else:
                msg = "Invalid band number in band list: '%s'" % b
                ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                ex.setData(Control.EX_USER_ERROR_MSG, msg)
                self.logError(msg)
                raise ex
        return c


    def getDeviceReferences(self):
        self.ants = self._array.antennas()
        errMsg = ""
        wcaConstructors = self.getWCAConstructors()
        for i in range(len(self.ants)):
            ant = self.ants[i]
            self.logInfo("Getting references to FrontEnd, LPR, IFSwitch and WCAs in antnena %s" % (ant))
            try:
                self.fes.append(CCL.FrontEnd.FrontEnd(ant))
                self.lprs.append(CCL.LPR.LPR(ant))
                lprState = self.lprs[i].getHwState()
                if lprState != Control.HardwareDevice.Operational:
                    if len(errMsg) > 0: errMsg += ", "
                    errMsg += "LPR of antenna %s is not Operational (it is in state %s)" % (ant, str(lprState))
                self.ifss.append(CCL.IFSwitch.IFSwitch(ant))
                ifsState = self.ifss[i].getHwState()
                if ifsState != Control.HardwareDevice.Operational:
                    if len(errMsg) > 0: errMsg += ", "
                    errMsg += "IFSwitch of antenna %s is not Operational (it is in state %s)" % (ant, str(ifsState))
                self.wcas.append([])
                for constructor in wcaConstructors:
                    wca = constructor(ant)
                    wcaState = wca.getHwState()
                    if wcaState != Control.HardwareDevice.Operational:
                        if len(errMsg) > 0: errMsg += ", "
                        errMsg += "%s of antenna %s is not Operational (it is in state %s)" % (wca.__class__.__name__, ant, str(wcaState))
                    self.wcas[i].append(wca)
            except:
                if len(errMsg) > 0: errMsg += ", "
                errMsg += "Error getting a reference to FE devices of antenna %s" % (ant)
        if len(errMsg) > 0:
            ex = ControlExceptionsImpl.FatalExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, errMsg)
            self.logError(errMsg)
            raise ex


    def bandCheckAndPowerUp(self):
        bandsToUse = []
        for b in self.bandList:
            bandsToUse.append(PyDataModelEnumeration.PyReceiverBand.fromString("ALMA_RB_%02d" % b))
        self.logInfo("We want to use bands %s" % str(bandsToUse))
        available = self.getCommonAvailableBands()
        self.logInfo("Common Available bands: %s" % (str(available)))
        powered = self.getCommonPoweredBands()
        self.logInfo("Common Powered bands: %s" % (str(powered)))
        allPowered = self.getPoweredBandStats()
        self.logInfo("Number of antennas with each band powered: %s" % (str(allPowered)))
        errMsg = ""
        for b in bandsToUse:
            if b not in available:
                if len(errMsg) > 0: errMsg += ". "
                errMsg += "Required band %s is not available in all antennas" % (str(b))
        if len(errMsg) > 0:
            ex = ControlExceptionsImpl.FatalExImpl()
            ex.setData(Control.EX_USER_ERROR_MSG, errMsg)
            self.logError(errMsg)
            raise ex
        bandsToPower = []
        for b in bandsToUse:
            if b not in powered:
                bandsToPower.append(b)
        # starting point for bands to run check scans
        for b in bandsToPower:
            self.bandsToCheck.append(int(str(b).split('_')[2]))
        if len(bandsToPower) > 0:
            self.logInfo("Preparing to power up bands %s" % (str(bandsToPower)))
            powerOff = self.getPowerOffBands(bandsToUse)
            self.logInfo("Powering off bands %s, and powering up bands %s" % (str(powerOff), str(bandsToPower)))
            self._array.setPoweredBands(bandsToPowerUp=bandsToPower, bandsToPowerDown=powerOff)


    def determineBandsToCheck(self):
        if len(self.bandsToCheck) == len(self.bandList):
            self.logInfo("Will check all bands as all were just powered up")
            return
        if self.checkAllBands:
            self.logInfo("Will check all bands as checkAllBands option was specified")
            self.bandsToCheck = self.bandList
            return
        for i in range(len(self.bandList)):
            if self.bandList[i] in self.bandsToCheck:
                continue
            if self.bandList[i] == 2:
                # ICT-22953: The WCA2 LOPA monitoring is not meaningful so assume
                # for now that we need to check the band. When Band 2 is used for
                # routine operations we should revisit this as it will waste time.
                self.bandsToCheck.append(2)
                continue
            for j in range(len(self.ants)):
                wca = self.wcas[j][i]
                vd0 = vd1 = 0.0
                try:
                    vd0 = wca.GET_LO_PA_POL0_DRAIN_VOLTAGE()[0]
                    vd1 = wca.GET_LO_PA_POL1_DRAIN_VOLTAGE()[0]
                except BaseException as ex:
                    self.logException("Error reading LOPA Vd values for band %d in antenna %s" % (self.bandList[i], self.ants[j]), ex)
                    raise ex
                self.logInfo("LOPA Vd values for band %d in antenna %s: [%.5f, %.5f]" % (self.bandList[i], self.ants[j], vd0, vd1))
                if vd0 < self.LOPAVdThreshold or vd1 < self.LOPAVdThreshold:
                    self.logInfo("A Vd value is below the threshold of %.5f V, so band %d added to list to check" % (self.LOPAVdThreshold, self.bandList[i]))
                    self.bandsToCheck.append(self.bandList[i])
                    break
        self.logInfo("After checking WCA LOPA Vd values the following bands should be checked: %s" % (str(self.bandsToCheck)))
        

    def selectBand(self, band):
        port = band - 1
        for i in range(len(self.ants)):
            ant = self.ants[i]
            self.logInfo("Selecting band %d in FrontEnd of antenna %s" % (band, ant))
            self.fes[i].selectBand(band)
            curPort = self.lprs[i].GET_OPT_SWITCH_PORT()[0]
            if curPort != port:
                self.logWarning("LPR port in antenna %s is %d, but expected %d (band %d), correcting it. See ICT-8230." % (ant, curPort, port, band))
                self.lprs[i].SET_OPT_SWITCH_PORT(port)
                curPort = self.lprs[i].GET_OPT_SWITCH_PORT()[0]
            # IFS is a bit inconsistent, returning zero-based cartridge but using 1-based set method names
            curCart = self.ifss[i].GET_CARTRIDGE()[0]
            if curCart != port:
                self.logWarning("IFSwitch input in antenna %s is %d, but expected %d (band %d), correcting it. See ICT-8230." % (ant, curCart, port, band))
                getattr(self.ifss[i], "SET_CARTRIDGE%d" % (band))()
                curCart = self.ifss[i].GET_CARTRIDGE()[0]
            self.logInfo("ant %s LPR port is %d, IFSwitch cartridge is %d" % (ant, curPort, curCart))


    def doDummyScans(self):
        for ss in self.dummyScanSpectralSpecs:
            try:
                dummyCal = Observation.CalTarget.CalTarget(SubscanFieldSource = self._srcPointFocus,
                            SpectralSpec = ss,
                            SubscanDuration = self.dummySubscanDuration,
                            IntegrationTime = self.dummySubscanDuration,
                            ScanIntentData = PyDataModelEnumeration.PyScanIntent.OBSERVE_TARGET)
                dummyCal.setWVRCalReduction(False)
                self.logInfo('Executing dummy scan on ' + self._srcPointFocus.sourceName + '...')
                dummyCal.execute(self._obsmode)
                self.logInfo('Completed dummy scan on ' + self._srcPointFocus.sourceName)
            except BaseException as ex:
                print(ex)
                msg = "Error executing dummy scan on source %s" % self._srcPointFocus.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex


    def doAtmCals(self):
        for ss in self.atmCalSpectralSpecs:
            try:
                atm = Observation.AtmCalTarget.AtmCalTarget(self._srcPointFocus, ss, doHotLoad=True)
                atm.setOnlineProcessing(True)
                atm.setDoZero(True)
                atm.setSubscanDuration(self.atmSubscanDuration)
                atm.setIntegrationTime(1.5)
                atm.setDataOrigin('TOTAL_POWER')
                atm.setWVRCalReduction(False)
                atm.setApplyWVR(False)
                # TODO: this should be band dependent
                atm.setBBLevel(-5.0)
                # Currently we need to set a reference source to use an offset.
                atm._referenceSource=atm._source
                atm._referenceOffset=CCL.SourceOffset.stroke(self.atmAzOffsetRad,0,0,0,Control.HORIZON)
                self.logInfo('Executing AtmCal on ' + self._srcPointFocus.sourceName + '...')
                atm.execute(self._obsmode)
                self.logInfo('Completed AtmCal on ' + self._srcPointFocus.sourceName)
            except BaseException as ex:
                print(ex)
                msg = "Error executing AtmCal on source %s" % self._srcPointFocus.sourceName
                self.logError(msg)
                self.closeExecution(ex)
                raise ex
 



obs = ObsCalBandCheckout()
obs.parseOptions()
#obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.bandCheckAndPowerUp()
    obs.getDeviceReferences()
    obs.determineBandsToCheck()
    obs.selectBand(obs.bandList[0])
    obs.generateTunings()
    obs.findPointFocusSource()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.doDummyScans()
obs.doAtmCals()
obs.closeExecution()

