#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: CalTargetSurvey.py 210969 2014-11-09 03:20:57Z nphillip $"
#
# who       when      what
# --------  --------  ----------------------------------------------
# jkern  2009-09-18  created
#

#************************************************************************
#   NAME CalTargetSurvey
# 
#   SYNOPSIS This script is intended to work for observatory modes where
#            the goal is to observe a set of calibration targets.  We
#            expect to find no science targets in this script!  Phase
#            calibrators are observed as needed if they are defined in a
#            group.
#
#            The first group is a calibration group which will execute
#            one of any type of calibration defined.
#
#
#------------------------------------------------------------------------
#


import sys
import Control
import Observation.SchedulingBlock
from CCL.Global import *
from CCL.ObservingModeBase import SBOutOfTime
from Observation.Global import OperatorLogger

logger = OperatorLogger()
sb     = Observation.SchedulingBlock.SchedulingBlock()

array  = getArray()
array.beginExecution()


try:
    repeatCount = int(sb.getExpertParameter('RepeatCount'))
except KeyError:
    repeatCount = 1
    pass

try:
    elLimit = sb.getExpertParameter("ElLimit")
except KeyError:
    elLimit = None
    pass

#    print "Error: To run this Observing mode you must specify " + \
#          "the expert mode paramter 'Cadence'"
#    array.endExecution(Control.FAIL, "Cadence parameter must be specified")
#    sys.exit()

obsmode = array.getInterferometryObservingMode()
obsmode.setMaximumSchedBlockTime(sb.getMaximumExecutionTime())
#
# DEBUG
#
#obsmode.setSignalLevelsNextTarget()
#
# DEBUG
#


if elLimit is not None and elLimit>=0:
    obsmode.setElevationLimit(elLimit)

groupList = sb.getGroupList()
#
# DEBUG
#
logger.logInfo('DOING CalTargetSurvey.py with ALMA-RELEASE-B VERSION')
#
# DEBUG
#

try:
    obsmode.executeGroupCalibrators(groupList[0])

    for group in groupList[1:]:
        # The primary phase calibrator is assumed to be the one with the
        # shortest cycle time in the group.  We ensure that this is always
        # observed to bracket the group.
        primaryPhaseCal = None
        for target in group.getPhaseCalTargetList():
            if primaryPhaseCal is None or \
                   target.getCycleTime() < primaryPhaseCal.getCycleTime():
                primaryPhaseCal = target

        if primaryPhaseCal is not None:
            obsmode.addTargetToCleanupList(primaryPhaseCal)
        
        targetListVector = [group.getPointingCalTargetList(),
                            group.getFocusCalTargetList(),
                            group.getDelayCalTargetList(),
                            group.getAmplitudeCalTargetList(),
                            group.getBandpassCalTargetList()]

        for idx in range(repeatCount):
            for targetList in targetListVector:
                for target in targetList:
                    target.execute(obsmode)
                    obsmode.executeFromTargetList(group.getPhaseCalTargetList())
                            
        # Reset the elevation limit
        obsmode.resetLimits()
                    

        # For a calibrator survey we want to make sure that each
        # group is bracketed by the primary phase calibrator
        obsmode.executeCleanupList()

except SBOutOfTime:
    # Ok there should be just enough time for the cleanup list now
    obsmode.executeCleanupList()
    
array.endExecution(Control.SUCCESS, "Successful Completion")    
            
#
# ___oOo___
