#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: AmpliCalSurvey.py 210969 2014-11-09 03:20:57Z nphillip $"
#
# who       when      what
# --------  --------  ----------------------------------------------
# scorder   2011-04-22  created

#************************************************************************
#   NAME AmpliCalTargetSurvey
# 
#   SYNOPSIS
#
#
#
#------------------------------------------------------------------------
#

primaryFluxCals         = ['mercury','venus','mars','callisto','io','europa','ganymede','titan','uranus','neptune','pluto','pallas','juno','vesta','ceres']

### Setting up for determining if the source is occulted by the planet but not implemented
### Also need to consider if the other moons are also in the beam.  Perhaps the dictionary structure should be reversed and assigned by parent body
sourceObstruction       = {'callisto' : 'jupiter' , 'ganymede' : 'jupiter' , 'io' : 'jupiter' , 'europa' : 'jupiter' , 'titan' : 'saturn'}

from types import MethodType
import sys,copy,math
import Control
import Observation.SchedulingBlock
from CCL.Global import *
from CCL.ObservingModeBase import SBOutOfTime
from Observation.AtmCalTarget import AtmCalTarget
from Observation.PointingCalTarget import PointingCalTarget
from Observation.BandpassCalTarget import BandpassCalTarget
import CCL.APDMSchedBlock
from Observation.TargetUtility import *
from Observation.Global import *
from CCL.logging import getLogger

def sortTargetListByBrightness(self,targetList):
    bandDefinitions = { 1  : [31.3e9, 45e9  ],
                        2  : [67e9  , 90e9  ],
                        3  : [84e9  , 116e9 ],
                        4  : [125e9 , 163e9 ],
                        5  : [163e9 , 211e9 ],
                        6  : [211e9 , 275e9 ],
                        7  : [275e9 , 370e9 ],
                        8  : [385e9 , 500e9 ],
                        9  : [602e9 , 720e9 ],
                        10 : [787e9 , 950e9]
                                                    }
    obsFreq = targetList[0]._spectralSpec.getMeanFrequency()
    for i in list(bandDefinitions.keys()) :
        if ((obsFreq <= bandDefinitions[i][1]) and (obsFreq >= bandDefinitions[i][0])) : band = i
    fLower, fUpper = bandDefinitions[band]

    class measureStructure:
        def __init__(self,target=None,date=None,flux=None) :
            self.name = target
            self.date = date
            self.flux = flux

    measures = []
    for i in targetList :
        src = i.getSource()
        measurements = self.wrapSearch(name=str(src.sourceName),fLower=fLower,fUpper=fUpper,sortBy='date_observed')
        if measurements != []:
            m = measureStructure(target = i,date = measurements[-1]['date_observed'].value,flux = measurements[-1]['flux'])
        else :
            m = measureStructure(target = i,flux=0.0)
        measures.append(m)
    return sorted(measures,key=lambda m: m.flux,reverse=True)

CalibratorCatalog.sortTargetListByBrightness = MethodType(sortTargetListByBrightness,None,CalibratorCatalog)

sb = Observation.SchedulingBlock.SchedulingBlock()

array = getArray()
array.beginExecution()
logger = getLogger()
# should work in 8.0.3 as well:
try:
    CCLVersion = getCCLVersion()
except:
    CCLVersion = '8.0.3 or older'

logger.logInfo('Running under CCL version %s' % CCLVersion)

###
### Parsing the expert parameters
###

try:
    repeatCount = int(sb.getExpertParameter('RepeatCount'))
except KeyError:
    repeatCount = 1
    pass
logger.logInfo('Repeat count set to %d ' % repeatCount)

try:
    pointingSubscanDuration = int(sb.getExpertParameter('PointingSubscanDuration'))
except KeyError:
    pointingSubscanDuration = 10
logger.logInfo('PointingSubscanDuration set to %f ' % pointingSubscanDuration)

try:
    atmSubscanDuration = int(sb.getExpertParameter('AtmSubscanDuration'))
except KeyError:
    atmSubscanDuration = 2
logger.logInfo('AtmSubscanDuration set to %f ' % atmSubscanDuration)
    
try:
    elLimit = sb.getExpertParameter("ElLimit")
except KeyError:
    elLimit = None
    pass
logger.logInfo('Elevation limit set to %s ' % elLimit)

try:
    maxTargets = int(sb.getExpertParameter('MaxTargets'))
except KeyError:
    maxTargets = 100
    pass
logger.logInfo('MaxTargets count set to %d ' % maxTargets)

obsmode = array.getInterferometryObservingMode()
obsmode.setMaximumSchedBlockTime(sb.getMaximumExecutionTime())

if elLimit is not None:
    obsmode.setElevationLimit(elLimit)

try:
    targetList = [i for i in sb.getAmplitudeCalTargetList()]
    # generate bandpass targets first
    cat = CalibratorCatalog()
    sortedMeasures = cat.sortTargetListByBrightness(targetList)
        
    brightestTarget = sortedMeasures[0].name
    logger.logInfo('Bandpass target will be %s (%s Jy)' %(brightestTarget.getSource().sourceName,
                                                          sortedMeasures[0].flux))
    # could write a setBandpassCalReduction(True) ...
    bandpassCalTarget = BandpassCalTarget(brightestTarget._source,brightestTarget._spectralSpec)
    bandpassCalTarget.setAmpliCalReduction(True)
    targetList.remove(brightestTarget)
    targetList.insert(0, bandpassCalTarget)
    targetList.append(bandpassCalTarget)
    # fails ? obsmode.addTargetToCleanUpList(bandpassCalTarget)
    ###
    ### Looping through the target list to dynamically generate the required targets
    ###
    orderedTargetList = []
    for amplitudeCalTarget in targetList :
        src = amplitudeCalTarget.getSource()
        spectralSpec = amplitudeCalTarget.getSpectralSpec()
        meanFreq = spectralSpec.getMeanFrequency()
        logger.logInfo('Mean Frequency is %f '%meanFreq)
        pntFreq = 230.538e9
        baseBandFreqs = [229.6005e9,231.4755e9,244.6005e9,246.4755e9]
        baseBandNames = ['BB_1','BB_2','BB_3','BB_4']
        chanAverageList = [[[6, 114]], [[6, 114]], [[6, 114]], [[6, 114]]]
        # Trick until we patch the simulator itself...
        # It looks like passing the obsmode in 8.1.0 starts a tuning to that band.
        if simulatedArray() or (CCLVersion >= '8.1.0'):
            spectralSpecPoint = CCL.SpectralSpec.SpectralSpec().CreateBLCorrelatorSpec(\
            corrMode='TDM',channelAverage=1.0,integration=5.0,channelAverageList=chanAverageList,
            baseband=baseBandNames,bwd=1,frequency=pntFreq,pol='2',
            basebandFrequency=baseBandFreqs)
            spectralSpecPoint.FrequencySetup.receiverBand = str('ALMA_RB_06')
        else:
            spectralSpecPoint = CCL.SpectralSpec.SpectralSpec().CreateBLCorrelatorSpec(\
            corrMode='TDM',channelAverage=1.0,integration=5.0,channelAverageList=chanAverageList,
            baseband=baseBandNames,bwd=1,frequency=pntFreq,pol='2',
            basebandFrequency=baseBandFreqs,
            obsMode=obsmode)
        ###
        ### Setting the subscan durations to multiples of the integration duration based on expert parameters
        ###
        intDur             = spectralSpec.getCorrelatorConfiguration().integrationDuration.get()
        pointingSubscanDur = max(intDur,round(pointingSubscanDuration/intDur)*intDur)
        atmSubscanDur      = max(intDur,round(atmSubscanDuration/intDur)*intDur)
        ###
        ### Setting up atmCalTarget list
        ###
        atmCalTarget = AtmCalTarget(src,spectralSpec,doHotLoad=True,doZero=False,DataOrigin='FULL_RESOLUTION_AUTO')
        atmCalTarget.setOnlineProcessing(True)
        atmCalTarget.setSubscanDuration(atmSubscanDur)
        atmCalTarget.setIntegrationTime(1.0)
        atmCalTarget.setWVRCalReduction(False)
        # Setting reference positions
        refSrc = copy.copy(src)
        refPos = CCL.APDMSchedBlock.Reference()
        refPos.setCoordinates('120 arcsec',0,system='azel')
        refPos.subScanDuration.set(atmSubscanDur)
        refPos.integrationTime.set(atmSubscanDur)
        refPos.cycleTime.set('1s')
        src.Reference = [refPos]
        ###
        ### Setting up the pointing cal target
        ###
        if meanFreq < 275e9 : pointingCalTarget = PointingCalTarget(src,spectralSpec)
        else : pointingCalTarget = PointingCalTarget(src,spectralSpecPoint)
        pointingCalTarget.setPointingMethod('FIVE_POINT')
        pointingCalTarget.setSubscanDuration(pointingSubscanDur)
        pointingCalTarget.setDataOrigin('CHANNEL_AVERAGE_CROSS')
        ###
        ### Forcing only the AtmCalTarget to use the reference
        ###
        pointingCalTarget.setUseReferencePosition(False)
        amplitudeCalTarget.setUseReferencePosition(False)
        atmCalTarget.setUseReferencePosition(True)
        ###
        ### Making pointing cal target list and associating the 
        ### atmCals with the amplitude cal targets.
        ### Eventually would be nice to associate a list
        ###
        amplitudeCalTarget.setAssociatedCalTarget(atmCalTarget)
        if meanFreq > 116e9 : orderedTargetList.append(pointingCalTarget)
        orderedTargetList.append(amplitudeCalTarget)

###
### Commenting out because I don't think we need this and we may be
### pressed for time
###
#    for target in amplitudeCalTargets :
#        src = target.getSource()
#        if  src.sourceName.lower() in primaryFluxCals :
#            obsmode.addTargetToCleanUpList(target)
    
    targetCount = 0
    for count in range(repeatCount) :
        for target in orderedTargetList :
            if target.isObservable(obsmode)  and targetCount < maxTargets:
                src = target.getSource()
                elongation = getSourceElongation(src)
                if ((elongation is None) or (elongation > math.radians(1.5/60.0))) :
                    targetCount += 1
                    logger.logInfo('Observe Target %d: %s' % (targetCount,
                                                              target.getSource().sourceName))
                    target.execute(obsmode)
#        obsmode.executeFromTargetList(orderedTargetList,minimumNumber=len(orderedTargetList))
    obsmode.resetLimits()
    obsmode.executeCleanupList()

except SBOutOfTime:
    # Ok there should be just enough time for the cleanup list now
    obsmode.executeCleanupList()
    
array.endExecution(Control.SUCCESS, "Successful Completion")    
            
#
# ___oOo___
