#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalOffPointCheck.py 247920 2017-08-08 15:07:12Z ahirota $"

# Based on ObsCalGrids.py. Things in common with ObsCalGrids should be 
# moved to a centralized place afterward.

#
# forcing global imports is due to an OSS problem
#
global sys
import sys
global math
import math
global datetime
import datetime
global re
import re
global os
import os
global traceback
import traceback


global CCL
import CCL.Global
import CCL.FieldSource

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global AcsutilPy
import AcsutilPy.FindFile

global Observation
import Observation.AtmCalTarget
import Observation.DelayCalTarget
import Observation.PointingCalTarget
import Observation.FocusCalTarget
import Observation.PhaseCalTarget
import Observation.BandpassCalTarget
import Observation.ObsCalBase
import Observation.ObsCalSources
import Observation.ShadowingChecker
import Observation.CalibratorCatalog


global specLineFreqs
specLineFreqs = {
    'sio(2-1)' : 86.2434300e9, # v=1,J=2-1)
    'hcn(1-0)' : 88.6318473e9, # HCN(v=0,J=1-0, triplet F=2-1,1-1,0-1, centered on 2-1)
    'hco+(1-0)': 89.1885260e9, # HCO+(v=0,J=1-0)
    'cs(2-1)'  : 97.9809530e9, # CS(v=0,J=2-1)
    'sis(6-5)' : 105.0592080e9, # SiS(v=0,J=6-5)
    'ch3oh107' : 107.0137700e9, # CH3OH(vt=0,3(1,3)-4(0,4)
    'c18o(1-0)': 109.7821760e9, # C18O(v=0,J=1-0)
    '13co(1-0)': 110.2013540e9, # 13CO(v=0,J=1-0)
    'cn113'    : 113.4909820e9, # CN(v=0,N=1-0 with several transitions)
    'co(1-0)'  : 115.2712020e9, # CO(1-0)
    'sio(3-2)' : 129.363359e9, #SiO(3-2) v=1
    'cs(3-2)'  : 146.969033e9, # CS(3-2)
    'hcn(2-1)' : 177.26111e9, # HCN(2-1)
    'sio(5-4)' : 215.5960680e9, # SiO)v=1,J=5-4)
    'c18o(2-1)': 219.5603600e9, # C18O(v=0,J=2-1)
    '13co(2-1)': 220.3986800e9, # 13CO(v=0,J=2-1)
    'co(2-1)'  : 230.5380000e9, # CO(v=0,J=2-1)
    'hcn(3-2)' : 265.8864300e9, # HCN(v=0,J=3-2)
    'hco+(3-2)': 267.5576300e9, # HCO+(v=0,J=3-2)
    'hnc(3-2)' : 271.9811400e9, # HNC(v=0,J=3-2)
    'co(3-2)'  : 345.7959900e9, # CO(v=0,J=3-2)
    'hcn(4-3)' : 345.5054700e9, # HCN(v=0,J=4-3)
    'hco+(4-3)': 356.7342400e9, # HCO+(v=0,J=4-3)
    'co(4-3)'  : 461.040768e9, # CO(v=0,J=4-3)
    'ci(1-0)'  : 492.160700e9, # CI(v=0,J=1-0)
    'hcn(7-6)' : 620.3041000e9, # HCN(v=0,J=7-6)
    'hco+(7-6)': 624.2084600e9, # HCO+(v=0,J=7-6)
    'h2o(1,1)' : 658.0065500e9, # H2O(v=1,1(1,0)-1(0,1))
    'co(6-5)'  : 691.4730800e9, # CO(v=0,J=6-5)
    'co(7-6)'  : 806.6518000e9, # CO(v=0,J=7-6)
    'n2h+(5-4)': 465.824780e9,  # N2H+(v=0,J=5-4)
}


class ObsCalOffPointCheck(Observation.ObsCalBase.ObsCalBase):

    # azoffset = math.radians(150.0/3600.0)

    options = [
        Observation.ObsCalBase.scriptOption("PointingSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("AtmSubscanDuration", float, 5.76),
        # Observation.ObsCalBase.scriptOption("SBRSubscanDuration", float, 5.76),
        # Observation.ObsCalBase.scriptOption("FocusSubscanDuration", float, 5.76),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.192),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 2.88),
        # Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.096),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("maxShadowFraction", float, 0.3),
        Observation.ObsCalBase.scriptOption("band", int, 3),
        Observation.ObsCalBase.scriptOption("subscanDuration", float, 20.),
        Observation.ObsCalBase.scriptOption("integrationTime", float, 60.),
        Observation.ObsCalBase.scriptOption("sourceListFile", str, ""),
        Observation.ObsCalBase.scriptOption("logFile", str, ""),
        Observation.ObsCalBase.scriptOption("obsLine", str, "co(1-0)"),
        Observation.ObsCalBase.scriptOption("siteLatitude", float, -23.0231944),
        Observation.ObsCalBase.scriptOption("noPointing", bool, False),
        Observation.ObsCalBase.scriptOption("onSourceATM", bool, True),
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self._reverseSpecs = False
        antennas = self._array.antennas()
        self._shadowingChecker = Observation.ShadowingChecker.ShadowingChecker(useSSRLogger=True,
                                                                               usedAntennaList=antennas)

    def parseOptions(self):
        # self.repeatCount             = self.args.RepeatCount
        self.pointingSubscanDuration = self.args.PointingSubscanDuration
        self.atmSubscanDuration      = self.args.AtmSubscanDuration
        # self.sbrSubscanDuration      = self.args.SBRSubscanDuration
        # self.focusSubscanDuration    = self.args.FocusSubscanDuration
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.subscanDuration         = self.args.subscanDuration
        self.integrationTime         = self.args.integrationTime
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        self.band                    = self.args.band
        self.obsLine                 = self.args.obsLine
        self.sourceListFile          = self.args.sourceListFile.strip()
        self.logFile                 = self.args.logFile.strip()
        self.maxShadowFraction       = self.args.maxShadowFraction
        self.siteLatitude            = self.args.siteLatitude
        self.noPointing              = self.args.noPointing

        if self.band < 1 or self.band > 10:
            raise Exception("Invalid band number specified: '%s'" % self.band)

        if self.obsLine in specLineFreqs:
            freq = specLineFreqs[self.obsLine]
        else:
            try:
                freq = float(self.obsLine) * 1.0e9
            except:
                msg = "Unrecognized line type [%s]" % (self.obsLine)
                raise self.getIllegalParameterError(msg)
        self.freq = freq

        self.logInfo("Observing frequency = %.3f [GHz]" % (self.freq))
        self.logInfo("Observing band      = %s" % (self.band))

    # Same as ObsCalGrid.py
    def generateTunings(self):
        corrType = self._array.getCorrelatorType()

        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.band,
                intent = "interferometry_continuum",
                corrType = corrType,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)

        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % (self.band)
        self._calSpectralSpecs = []

        ss = self._tuningHelper.GenerateSpectralSpec(band=self.band,
                                                     corrMode='FDM',
                                                     bbFreqs=[self.freq],
                                                     autoOnly=True,
                                                     bwd=4.,
                                                     intent = "calsurvey",
                                                     corrType = corrType,
                                                     dualMode = True,
                                                     pol='2',
                                                     dump = self.dumpDuration,
                                                     channelAverage = self.channelAverageDuration,
                                                     integration = self.integrationDuration,
                                                     tpSampleTime = self.tpIntegrationDuration)

        if self.obsLine == 'co(1-0)':
            cFreq0 = 115.0e9
            for bbs in ss.FrequencySetup.BaseBandSpecification:
                offset = bbs.centerFrequency.get() - cFreq0
                bbs.centerFrequency.set(cFreq0)
                if ss.BLCorrelatorConfiguration:
                    bbc = ss.BLCorrelatorConfiguration.BLBaseBandConfig[0]
                    spw = bbc.BLSpectralWindow[0]
                    spw.centerFrequency.set(3e9 + offset)
                elif ss.ACACorrelatorConfiguration:
                    abc = ss.ACACorrelatorConfiguration.ACABaseBandConfig[0]
                    spw = abc.ACASpectralWindow[0]
                    spw.centerFrequency.set(3e9 + offset)
                elif ss.ACASpectrometerCorrelatorConfiguration:
                    abc = ss.ACASpectrometerCorrelatorConfiguration.ACABaseBandConfig[0]
                    spw = abc.ACASpectralWindow[0]
                    spw.centerFrequency.set(3e9 + offset)
                else:
                    msg = 'The observing array does not have correlator configured'
                    raise self.getIllegalParameterError(msg)

        if ss.BLCorrelatorConfiguration:
            for bbc in ss.BLCorrelatorConfiguration.BLBaseBandConfig:
                for spw in bbc.BLSpectralWindow:
                    spw.windowFunction = u'HANNING'
                    spw.spectralAveragingFactor = 2
        elif ss.ACACorrelatorConfiguration:
            for abc in ss.ACACorrelatorConfiguration.ACABaseBandConfig:
                for spw in abc.ACASpectralWindow:
                    spw.windowFunction = u'HANNING'
                    spw.spectralAveragingFactor = 2
        elif ss.ACASpectrometerCorrelatorConfiguration:
            for abc in ss.ACASpectrometerCorrelatorConfiguration.ACABaseBandConfig:
                for spw in abc.ACASpectralWindow:
                    spw.windowFunction = u'HANNING'
                    spw.spectralAveragingFactor = 2
        else:
            msg = 'The observing array does not have correlator configured'
            raise self.getIllegalParameterError(msg)

        self.logInfo("%s" % (ss.toDOM().toprettyxml()))
        ss.name = "Band %d calsurvey" % (self.band)
        self._calSpectralSpecs.append(ss)

    def getIllegalParameterError(self, msg):
        ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
        ex.setData(Control.EX_USER_ERROR_MSG, msg)
        return ex

    def _readFile(self, fPath):
        if not os.path.isfile(fPath):
            msg = "Unable to find the specified file '%s'" % (fPath)
            raise self.getIllegalParameterError(msg)
        with open(fPath) as f:
            lines = [line.strip() for line in f.readlines()]
        return lines

    def readObservingLog(self):
        import collections
        observedCounts = collections.defaultdict(int)
        if self.logFile == "":
            msg = "Log file for recording source executions not specified"
            self.logWarning(msg)
            return observedCounts

        pat_sep = re.compile("[ \t]+")
        if not os.path.isfile(self.logFile):
            with open(self.logFile, "w") as f:
                pass
        lines = self._readFile(self.logFile)
        for iLine, line in enumerate(lines, 1):
            if len(line) == 0 or line.startswith("#"):
                continue
            tokens = pat_sep.split(line)
            if len(tokens) != 4:
                msg = "[%s:L%d] Illegal line in the log file ('%s')" % (self.logFile, iLine, line)
                raise self.getIllegalParameterError(msg)

            uid = tokens[0]
            ra = tokens[1]
            dec = tokens[2]
            offPointName = tokens[3]

            if not Observation.Global.simulatedArray() and uid == 'uid://A00/X00/X00':
                # This is a dummy UID used by OSS simulations.
                continue
            observedCounts[offPointName] += 1
        return observedCounts

    def readSources(self):
        pat_sep = re.compile("[ \t]+")

        if self.sourceListFile == "":
            msg = "Both 'sources' and 'sourceListFile' options are not specified"
            raise self.getIllegalParameterError(msg)

        sourceDict = dict()
        observedCounts = self.readObservingLog()
        lines = self._readFile(self.sourceListFile)
        for iLine, line in enumerate(lines, 1):
            if len(line) == 0 or line.startswith("#"):
                continue
            tokens = pat_sep.split(line)
            if len(tokens) != 5:
                msg = "[%s:L%d] Unexpected format ('%s')" % (self.sourceListFile, iLine, line)
                raise self.getIllegalParameterError(msg)

            offPointName = tokens[0]
            ra = float(tokens[1])
            dec = float(tokens[2])
            offRa = float(tokens[3])
            offDec = float(tokens[4])

            if offPointName in sourceDict:
                values = (self.sourceListFile, iLine, offPointName)
                msg = "[%s:L%d] Off-position name '%s' is not unique" % values
                raise self.getIllegalParameterError(msg)

            if offPointName in observedCounts and observedCounts[offPointName] >= 1:
                continue

            values = (offPointName, ra, dec, observedCounts[offPointName])
            msg = "%s ra=%9.4f dec=%9.4f execCount=%2d" % values
            self.logInfo("[readSources] %s" % msg)

            sourceDict[offPointName] = dict(approxRa=ra, approxDec=dec,
                                            offRa=offRa, offDec=offDec,
                                            fs=None)

        self.logInfo("[readSources] %s" % "Read %d sources" % (len(sourceDict)))
        return sourceDict

    def pickupCalSources(self):
        self.sourceDict = self.readSources()
        # Estimate source flux for pointing/focus source for cal spectralspec
        # SPW center frequencies.
        cc = Observation.CalibratorCatalog.CalibratorCatalog('observing')
        cc.estimateSourceFlux(self._srcPointFocus, self._calSpectralSpecs,
                              setSourceProperties=True)

    # Same as ObsCalGrid.py
    def orderedSpecs(self):
        ret = self._calSpectralSpecs
        if self._reverseSpecs:
            ret = reversed(self._calSpectralSpecs)
        self._reverseSpecs = not self._reverseSpecs
        return ret

    # Same as ObsCalGrid.py
    def doPointing(self):
        if self.noPointing:
            return
        try:
            pointingCal = Observation.PointingCalTarget.PointingCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            pointingCal.setSubscanDuration(self.pointingSubscanDuration)
            pointingCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            # if not self.pipelineFriendly:
            #   pointingCal.setDelayCalReduction(True)
            self.logInfo('Executing PointingCal on ' + self._srcPointFocus.sourceName + '...')
            pointingCal.execute(self._obsmode)
            self.logInfo('Completed PointingCal on ' + self._srcPointFocus.sourceName)
            result = pointingCal.checkResult(self._array)
            self.logInfo("Result is: %s" % str(result))
            if len(result) > 0:
                for key in list(result.keys()):
                    self.logInfo("Found solution for %s using polarization(s) %s" %
                            (key, result[key]))
                pointingCal.applyResult(self._obsmode, result)
            else:
                if not "OSS" in self._array._arrayName:
                    raise Exception("No pointing results!")
        except BaseException as ex:
            print(ex)
            msg = "Error executing pointing on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def doATMCal(self, src, ss):
        atm = Observation.AtmCalTarget.AtmCalTarget(src, ss, doHotLoad=True)
        atm.setOnlineProcessing(True)
        atm.setDataOrigin('FULL_RESOLUTION_AUTO')
        atm.setDoZero(False)
        atm.setSubscanDuration(self.atmSubscanDuration)
        atm.setIntegrationTime(1.5)
        atm.setWVRCalReduction(True)
        # Applying the results takes a while with lots of
        # antennas, so until we use online WVR, don't bother
        atm.setApplyWVR(False)
        atm.convertAbsRefToRel = True
        atm.prepareReferencePosition()

        if self.args.onSourceATM:
            # ICT-23915
            atm.setDoOnSource(True)

        # Automatically adjust reference position for this target.
        atm.tweakReferenceOffset()
        self.logInfo('Executing AtmCal on ' + src.sourceName + '...')
        atm.execute(self._obsmode)
        self.logInfo('Completed AtmCal on ' + src.sourceName)

    def doCalSource(self, src, intent):
        assert(intent == 'OnOff')
        import Observation.SSRTuning
        from Observation.OnOffTarget import OnOffTarget
        for ss in self.orderedSpecs():
            ssAtm = Observation.SSRTuning.generateAtmSpectralSpec(ss)
            try:
                self.doATMCal(src, ssAtm)

                target = OnOffTarget(src, ss,
                                     IntegrationTime=self.integrationTime * 2,
                                     SubscanDuration=self.subscanDuration)

                # from PyDataModelEnumeration import PyCalibrationDevice
                # self._obsmode.setCalibrationDevice(PyCalibrationDevice.NONE)
                sName = src.sourceName
                self.logInfo('Executing %sCal on %s ...' % (intent, sName))
                target.execute(self._obsmode)
                self.logInfo('Completed %sCal on %s' %  (intent, sName))

            except BaseException as ex:
                msg = "Error executing cal survey scans on source %s" % src.sourceName
                self.logException(msg, ex)
                self.closeExecution(ex)
                raise ex

    @staticmethod
    def calcAzEl(raRad, decRad, lstHour, siteLatitudeRad):
        # TODO: should be moved to a more appropriate place
        import numpy as np
        lst = np.radians((360.0/24.0) * lstHour)
        cosDec = np.cos(decRad)
        sinDec = np.sin(decRad)
        cosLat = np.cos(siteLatitudeRad)
        sinLat = np.sin(siteLatitudeRad)
        cosHa = np.cos(lst - raRad)
        sinHa = np.sin(lst - raRad)
        az = np.arctan2(-cosDec * sinHa,
                        cosLat * sinDec - sinLat * cosDec * cosHa)
        az = az % (2.0 * np.pi)
        el = np.arcsin(sinLat * sinDec + cosLat * cosHa * cosDec)
        return az, el

    @staticmethod
    def calcHAOffsetFromElevation(decRad, elRad, siteLatRad):
        # TODO: should be moved to a more appropriate place
        import numpy as np
        sinLat_ = np.sin(siteLatRad)
        cosLat_ = np.cos(siteLatRad)
        sinDec_ = np.sin(decRad)
        cosDec_ = np.cos(decRad)
        cosHA = (np.sin(elRad) - sinDec_ * sinLat_) / (cosLat_ * cosDec_)
        if np.abs(cosHA) > 1:
            return np.nan
        absHa = np.arccos(cosHA)
        return absHa

    def selectNextSources(self, N=3,
                          minEl=20., maxEl=87.,
                          preferredElMin=50.):
        import numpy as np
        import CCL.APDMSchedBlock
        import CCL.FieldSource

        srcNames = list(self.sourceDict.keys())
        lstHour = self.sourceHelper.getLSTHour()
        self.logInfo("LST = %.2f [hour]" % (lstHour))

        siteLatitudeRad = np.radians(self.siteLatitude)
        candidates = []
        for srcName in srcNames:
            aRa = self.sourceDict[srcName]["approxRa"]
            aDec = self.sourceDict[srcName]["approxDec"]
            aAz, aEl = self.calcAzEl(np.radians(aRa), np.radians(aDec),
                                     lstHour, siteLatitudeRad)
            aAz = np.degrees(aAz)
            aEl = np.degrees(aEl)
            self.logInfo("%-30s %7.2f %7.2f" % (srcName, aAz, aEl))

            if aEl < minEl or aEl > maxEl:
                continue
            fShadow = self._shadowingChecker.isBlocked(aAz, aEl,
                                                       returnFraction=True)
            aElMax = 90. - np.abs(aDec - self.siteLatitude)

            # Derive how long the source will be over the preferred lower
            # limit of the elevation. This one could be 'nan'.
            haOff = self.calcHAOffsetFromElevation(np.radians(aDec),
                                                   np.radians(preferredElMin),
                                                   siteLatitudeRad)
            timeToBeCaught = (aRa / 15. + haOff * (12. / np.pi)) - lstHour
            timeToBeCaught = timeToBeCaught % 24.
            if timeToBeCaught > 12:
                timeToBeCaught -= 24.

            # Derive how long the source will be over 20 degrees
            # This one also could be 'nan' .
            haOff = self.calcHAOffsetFromElevation(np.radians(aDec),
                                                   np.radians(20.),
                                                   siteLatitudeRad)
            durationOverElLimit = haOff * (12. / np.pi) * 2.

            score = 5.
            if fShadow > 0.:
                score = score / (fShadow / 0.03)
            if np.isnan(durationOverElLimit):
                score = -1.
            elif durationOverElLimit > 1.:
                score = score / durationOverElLimit ** 0.5
            if timeToBeCaught > -0.2:
                score = score / np.maximum(timeToBeCaught, 0.5)
            if aEl < preferredElMin:
                score = score * aEl / preferredElMin
            if aEl / aElMax < 0.4:
                score = -1.
            if fShadow > self.maxShadowFraction:
                score = -1.
            entry = [srcName, aAz, aEl, fShadow, durationOverElLimit, timeToBeCaught, score]
            candidates.append(entry)

        sortedCandidates = sorted(candidates, key=lambda c: c[-1], reverse=True)
        for srcName, aAz, aEl, fShadow, durationOverElLimit, timeToBeCaught, score in sortedCandidates:
            values = (srcName, aAz, aEl, fShadow, 20., durationOverElLimit, timeToBeCaught, score)
            msg = "[%s] (approx) (az,el)=(%3d,%3d) fShadow=%.1f d(El>%.1f)=%5.1f [hour]  remain=%4.1f [hour]  score=%4.2f" % values
            self.logInfo(msg)
        selectedSrcNames = [c[0] for c in sortedCandidates[:N] if c[-1] > 0]
        # Obtain FieldSource instances
        for srcName in selectedSrcNames:
            if self.sourceDict[srcName]["fs"]:
                continue
            # absolute Ra, Dec of the target coordinates (in degrees)
            ra = self.sourceDict[srcName]["approxRa"]
            dec = self.sourceDict[srcName]["approxDec"]
            # absolute Ra, Dec of the reference coordinates (in degrees)
            offRa = self.sourceDict[srcName]["offRa"]
            offDec = self.sourceDict[srcName]["offDec"]

            src = CCL.FieldSource.EquatorialSource(sourceName=srcName,
                                                   ra="%.15f deg" % ra,
                                                   dec="%.15f deg" % dec)
            refPos = CCL.APDMSchedBlock.Reference()
            refPos.setCoordinates("%.15f deg" % offRa, "%.15f deg" % offDec,
                                  system=u"ICRS", type=u"ABSOLUTE")
            # print refPos.toDOM().toprettyxml()
            refPos.integrationTime.set("4s")
            refPos.subScanDuration.set("2.016s")
            assert(len(src.Reference) == 0)
            src.Reference.append(refPos)

            self.sourceDict[srcName]["fs"] = src
        # Ignore sources which could not be resolved.
        selectedSrcNames = [sN for sN in selectedSrcNames if self.sourceDict[sN]["fs"]]
        return selectedSrcNames

    def writeLogEntry(self, srcName, ra, dec):
        if self.logFile == "":
            return
        self.logInfo("writeLogEntry(): adding record for '%s'" % srcName)
        oldUmask = os.umask(0)
        fd = os.open(os.path.expanduser(self.logFile), os.O_WRONLY | os.O_CREAT | os.O_APPEND, 0o666)
        fp = os.fdopen(fd, "a")
        msg = "%-21s %14.9f %14.9f %s\n" % (str(self.uid), ra, dec, str(srcName))
        fp.write(msg)
        fp.close()
        os.umask(oldUmask)

    def determinePreferredMinimumElevation(self):
        antennas = self._array.antennas()
        nCM = len([ant for ant in antennas if ant.startswith("CM")])
        if nCM > 6:
            preferredElMin = 50.
        else:
            preferredElMin = 40.
        msg = "%d CM antennas included in the array." % (nCM)
        msg += " Will Prefer sources over el=%d deg" % (preferredElMin)
        self.logInfo(msg)
        return preferredElMin

    def isObservable(self, srcName,
                     minEl=20.,
                     noShadowingCheck=False):
        import numpy as np
        src = self.sourceDict[srcName]["fs"]
        # Check elevation and shadowing
        az, el = Observation.ObsCalSources.ObsCalSources.sourceAzEl(src)
        az = np.degrees(az)
        el = np.degrees(el)
        fShadow = self._shadowingChecker.isBlocked(az, el,
                                                   returnFraction=True)
        isObs = True
        # TODO: Elevation should be made also by looking ahead by
        #       several minutes. As a temporal workaround, just add
        #       2.5 degree to minEl
        if el < (minEl + 2.5) or el > 87:
            isObs = False
        if fShadow > self.maxShadowFraction:
            isObs = False
            values = (src.sourceName, fShadow * 100)
            msg = "Will skip source '%s' as %.1f percent of antennas will suffer from shadowing" % values
            self.logInfo(msg)

        values = (srcName, az, el, minEl, fShadow, isObs)
        msg = "[%s] (az,el)=(%6.1f,%5.1f) minEl=%.1f fShadow=%.1f isObservable=%s" % values
        self.logInfo(msg)

        if not isObs:
            return False
        return True

    def doCalObservations(self):
        minEl = self._obsmode.getElevationLimit()
        minEl = math.degrees(CCL.SIConverter.toRadians(minEl))

        # Survey execution
        preferredElMin = self.determinePreferredMinimumElevation()
        nOnOffScansMade = 0
        while 1:
            srcNames = self.selectNextSources(N=3, minEl=minEl,
                                              preferredElMin=preferredElMin)
            if len(srcNames) == 0:
                break

            nObserved = 0
            for srcName in srcNames:
                if not self.isObservable(srcName, minEl=minEl):
                    continue
                src = self.sourceDict[srcName]["fs"]
                ra = self.sourceDict[srcName]["approxRa"]
                dec  = self.sourceDict[srcName]["approxDec"]

                self.doCalSource(src, "OnOff")
                self.writeLogEntry(srcName, ra, dec)
                self.sourceDict.pop(srcName)
                nOnOffScansMade += 1
                nObserved += 1

            if nObserved == 0:
                # Guard for introducing an infinite loop.
                msg = "Although %d sources picked, none of them are observable" % (len(srcNames))
                self.logInfo(msg)
                break

            if nOnOffScansMade >= 10:
                break

    def configureArray(self):
        # Call this after parsing command line options, to set manual mode array.
        # Note this runs before the logger is ready.
        arrayName = self.args.array
        # Override to add a support for OSS simulation
        if arrayName in ["ArrayOSS"]:
            from ObservingModeSimulation.ArraySimulator import SimulatedArray
            array_ = SimulatedArray()
            array_._arrayName = arrayName
            CCL.Global._array = array_
            CCL.Global.getArray = lambda : array_
            from Observation.AntennaPositionData import setupSimulatedTMCDBData
            setupSimulatedTMCDBData("tp", array=array_)
            array_.setCorrelatorType("ACA")
            ObsCalOffPointCheck.arrayAlreadyDefined = lambda x: False
            ObsCalOffPointCheck.configureArray = lambda x: (sys.stdout.write("%s\n" % x.args.array))
            return
        Observation.ObsCalBase.ObsCalBase.configureArray(self)


survey = ObsCalOffPointCheck()
survey.parseOptions()
survey.checkAntennas()
survey.startPrepareForExecution()
try:
    survey.generateTunings()
    survey.findPointFocusSource()
    survey.pickupCalSources()
except BaseException as ex:
    survey.logException("Error in methods run during execution/obsmode startup", ex)
    survey.completePrepareForExecution()
    survey.closeExecution(ex)
    raise ex


survey.completePrepareForExecution()
# survey.logInfo("Executing pointing...")
# survey.doPointing()
survey.logInfo("Executing Calibration observations...")
survey.doCalObservations()
survey.closeExecution()
