#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: FocusCalObservation.py 214237 2015-02-06 05:16:28Z nphillip $"

#
# forcing global imports is due to an OSS problem
#
global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.ObsCalBase
import Observation.PhaseCalTarget
import Observation.PrimaryBeamCalTarget
global ArraySplit
from Observation.ArraySplit import ArraySplit


class ObsCalAstroHolo(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("SubscanDuration", float, 4.608),
        Observation.ObsCalBase.scriptOption("PhaseSubscanDuration", float, 60.0),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.048),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.048),
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 1.536),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "2 deg"),
        Observation.ObsCalBase.scriptOption("band", int, 6),
        Observation.ObsCalBase.scriptOption("sourceName", str, ""),
        Observation.ObsCalBase.scriptOption("numMapSubscans", int, 24),
        Observation.ObsCalBase.scriptOption("numMapRepeats", int, 2),
        Observation.ObsCalBase.scriptOption("numOverallRepeats", int, 1),
        Observation.ObsCalBase.scriptOption("refAntGroup", int, 1),
        Observation.ObsCalBase.scriptOption("refAntList", str, ""),
        Observation.ObsCalBase.scriptOption("rasterPattern", bool, False),
        Observation.ObsCalBase.scriptOption("relativeOversampling", float, 1.0),
        Observation.ObsCalBase.scriptOption("frequency", float, -1.),
        Observation.ObsCalBase.scriptOption("E", float, -140.79),
        Observation.ObsCalBase.scriptOption("N", float, -5189.69),
        Observation.ObsCalBase.scriptOption("U", float, 321.36 + 7. - 0.15),
        Observation.ObsCalBase.scriptOption("pointingOffsetFile", str, ""),
        Observation.ObsCalBase.scriptOption("delayOffsetFile", str, ""),
        Observation.ObsCalBase.scriptOption("focusOffsetFile", str, ""),
    ]

    def __init__(self):
        Observation.ObsCalBase.ObsCalBase.__init__(self)
        self._srcPointFocus = None
        self.refAntennas = []

    def parseOptions(self):
        self.primaryBeamSubscanDuration = self.args.SubscanDuration
        self.phaseSubscanDuration       = self.args.PhaseSubscanDuration
        self.dumpDuration               = self.args.dumpDuration
        self.channelAverageDuration     = self.args.channelAverageDuration
        self.integrationDuration        = self.args.integrationDuration
        self.tpIntegrationDuration      = self.args.tpIntegrationDuration
        self.elLimit                    = self.args.ElLimit
        self.band                       = self.args.band
        self.userSpecifiedSource        = self.args.sourceName
        self.numMapSubscans             = self.args.numMapSubscans
        self.numMapRepeats              = self.args.numMapRepeats
        self.numOverallRepeats          = self.args.numOverallRepeats
        self.refAntGroup                = self.args.refAntGroup
        self.refAntList                 = self.args.refAntList
        self.rasterPattern              = self.args.rasterPattern
        self.relativeOversampling       = self.args.relativeOversampling

        self.frequency = self.args.frequency
        self.bPosE = self.args.E
        self.bPosN = self.args.N
        self.bPosU = self.args.U

        self.pointingOffsetFile = self.args.pointingOffsetFile
        self.delayOffsetFile = self.args.delayOffsetFile
        self.focusOffsetFile = self.args.focusOffsetFile

    def defineRefAnts(self):
        antennaList = self._array.antennas()
        self.logInfo("antennaList: %s" % str(antennaList))
        if len(self.refAntList) > 0:
             refAntennas = [self.refAntList.split(',')]
        else:
            parts = ArraySplit(antennaList).split()
            self.logInfo("parts: %s" % str(parts))
            if self.refAntGroup == 0:
                refAntennas = parts
            else:
                refAntennas = [parts[self.refAntGroup-1]]
        # Check to make sure the antennas are really in the array
        for ref in refAntennas:
            for antenna in ref:
                if not antenna in antennaList:
                    msg = "Antenna " + antenna + " is not in the array"
                    ex = ControlExceptionsImpl.IllegalParameterErrorExImpl()
                    ex.setData(Control.EX_USER_ERROR_MSG, msg)
                    self.logError(msg)
                    raise ex
        self.logInfo("refAntennas: %s" % str(refAntennas))
        self.refAntennas = refAntennas

    def generateTunings(self):
        corrType = self._array.getCorrelatorType()

        if self.frequency > 0:
            frequency_ = self.frequency
        else:
            frequency_ = None

        self._pointFocusSpectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.band,
                intent = "interferometry_continuum",
                pol = "4",
                frequency = frequency_,
                LO_offset_strategy = 2,
                corrType = corrType,
                # ICT-5415 Enabling dual mode causes problems for CLIC
                dualMode = False,
                dump = self.dumpDuration,
                # sameBBFreqs = sameBBFreqs,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration,
                tpSampleTime = self.tpIntegrationDuration)
        self._pointFocusSpectralSpec.name = "Band %d pointing/focus" % self.band

    def doPhaseCal(self):
        try:
            phaseCal = Observation.PhaseCalTarget.PhaseCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            phaseCal.setSubscanDuration(self.phaseSubscanDuration)
            phaseCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            phaseCal.setIntegrationTime(1.0)
            phaseCal.setDelayCalReduction(True)
            self.logInfo('Executing PhaseCal on ' + self._srcPointFocus.sourceName + '...')
            phaseCal.execute(self._obsmode)
            self.logInfo('Completed PhaseCal on ' + self._srcPointFocus.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing phase on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def doPrimaryBeam(self):
        try:
            primaryBeamCal = Observation.PrimaryBeamCalTarget.PrimaryBeamCalTarget(self._srcPointFocus, self._pointFocusSpectralSpec)
            primaryBeamCal.setSubscanDuration(self.primaryBeamSubscanDuration)
            primaryBeamCal.setNumSubscans(self.numMapSubscans)
            primaryBeamCal.setIntegrationTime(self.numMapRepeats * self.numMapSubscans * self.primaryBeamSubscanDuration - 1.0)
            primaryBeamCal.setDataOrigin('CHANNEL_AVERAGE_CROSS')
            primaryBeamCal._rasterPattern = self.rasterPattern
            primaryBeamCal._relativeOversampling = self.relativeOversampling
            for ref in self.refAntennas:
                self.logInfo("Setting Reference antennas to: " + str(ref))
                primaryBeamCal.setReferenceAntennas(ref)
                self.logInfo('Executing PrimaryBeamCal on ' + self._srcPointFocus.sourceName + '...')
                primaryBeamCal.execute(self._obsmode)
                self.logInfo('Completed PrimaryBeamCal on ' + self._srcPointFocus.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing primaryBeam on source %s" % self._srcPointFocus.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex

    def findPointFocusSource(self, useThisSource=None, **kwargs):
        import Observation.ObsCalBase
        if not self.userSpecifiedSource == "beacon" and \
            not useThisSource == "beacon":
            Observation.ObsCalBase.ObsCalBase.findPointFocusSource(self, **kwargs)
            return
        from Observation.ArtificialSourceHelper import ArtificialSourceHelper
        bPosENU = [self.bPosE, self.bPosN, self.bPosU]
        builder = ArtificialSourceHelper(bPosENU)
        builder.readOffsetsFromFile("pointing", self.pointingOffsetFile)
        builder.readOffsetsFromFile("delay", self.delayOffsetFile)
        # builder.readOffsetsFromFile("focus", self.focusOffsetFile)
        refAnt = builder.selectReferenceAntenna()
        src = builder.createFieldSource(referenceAntenna=refAnt)
        self.logInfo("beacon FieldSource: %s" % str(src.toxml()))
        self._srcPointFocus = src


obs = ObsCalAstroHolo()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.findPointFocusSource(minEl=30.0, useThisSource=obs.userSpecifiedSource)
    obs.defineRefAnts()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
for n in range(obs.numOverallRepeats):
    obs.logInfo("Running repeat %d of %d" % (n+1, obs.numOverallRepeats))
    obs.logInfo("Executing first a short phaseCal for QA of phase stability etc...")
    obs.doPhaseCal()
    obs.logInfo("Executing the primary beam scan(s)...")
    obs.doPrimaryBeam()
obs.closeExecution()

