#*******************************************************************************
# ALMA - Atacama Large Millimeter Array
# Copyright (c) UKATC - UK Astronomy Technology Centre, Science and Technology Facilities Council, 2011
# (in the framework of the ALMA collaboration).
# All rights reserved.
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#*******************************************************************************
# Python wrappers for ObservingTool actions.
#
# Author: amchavan, 06-Nov-2003
# $Id: ot.py,v 1.5.92.1 2009/03/12 10:30:05 hyatagai Exp $
#
# msc 2010-08: This is a library script that always gets executed
# when the OT is asked to run a batch script. This script used to
# define many functions, but they were now migrated into Java code.

from alma.obsprep.ot.scripting    import BatchOT


# The python variable "batchOT" is passed from the batch-mode launcher to all
# python scripts. It contains a reference to the batch-mode OT application that hosts this
# script and offers all of the API that used to be defined as python functions in this library.
#
# Note: There are other places that also execute python scripts, and each of them would need to
# pass that reference if their scripts want to store, retrieve, import, export, etc. To provide
# some better error diagnostics for such cases, it would be useful if scripts only accessed the
# variable through a getter function in this library, so the getter could print some informative
# message if the variable wasn't set. This is not compatible with some of our unit tests
# (apparently for scope reasons). So instead of a getter, we'll have a sanity checker function
# that can be invoked on a voluntary basis. If scripts directly go for the variable without sanity
# checking, there'll just be NameError with little indication about what went wrong.

def check_hosted_by_batchOT():
   try:
      batchOT
   except NameError:
      print 'Note: This script is not being run through "ALMA-OT -batch". This means some'
      print '      necessary setup for actions like store, retrieve, etc. may be missing.'

__version__ = "$Revision: 1.5.92.1 $"



##
## EVERYTHING BELOW HERE TO BE DELETED 
## (not doing it now because some unit tests still exercise this stuff)
##
from alma.obsprep.ot.scripting    import PyAdd
from alma.obsprep.ot.scripting    import PyImport
from alma.obsprep.ot.scripting    import PyExport
from alma.obsprep.ot.scripting    import PyNewProject
from alma.obsprep.ot.scripting    import PyStore
from alma.obsprep.ot.scripting    import PyRetrieve
from alma.obsprep.ot.scripting    import PyQuery
from alma.obsprep.ot.scripting    import PyGenerate
from alma.obsprep.ot.actions      import AddInteractor


#-----------------------------------------------------------------------
# Types of Business Objects for the add( boType, boParent) command
#-----------------------------------------------------------------------
SCHEDBLOCK      = AddInteractor.ObjectType.SCHEDBLOCK;
OBSUNITSET      = AddInteractor.ObjectType.OBSUNITSET;
FIELDSOURCE     = AddInteractor.ObjectType.FIELDSOURCE;
SCIENCEGOAL     = AddInteractor.ObjectType.SCIENCEGOAL;

__version__ = "$Revision: 1.5.92.1 $"


#-----------------------------------------------------------------------
# Imports a project
# @param  pathname   Pathname of the file to import. 
#                    Relative pathnames are interpreted relative to
#                    the ObservingTool runtime directory
# @return The imported project
#-----------------------------------------------------------------------
def importprj( pathname ):
    action = PyImport( pathname )       # Instanciate the action object
    action.perform()                    # Perform the action
    project = action.getObsProject()    # Retrieve the new ObsProject...
    return project                      # ...and return it


#-----------------------------------------------------------------------
# Exports a project
# @param  pathname   Pathname of the file to export. 
#                    Relative pathnames are interpreted relative to
#                    the ObservingTool runtime directory
# @return Nothing
#-----------------------------------------------------------------------
def exportprj( obsproject, pathname ):
    action = PyExport( pathname )  # Instanciate the action object
    action.setItemToExport( obsproject )
    action.perform()                    # Perform the action


#-----------------------------------------------------------------------
# Gets a project from the archive - by entity id
# @param  entityID   Entity ID of the desired project
# @return The imported project
#-----------------------------------------------------------------------
def retrieveprj( entityID ):
    otc = BatchOT()
    action = PyRetrieve( entityID, otc )     # Instanciate the action object
    action.perform()                    # Perform the action
    project = action.getObsProject()    # Retrieve the new ObsProject...
    return project                      # ...and return it


#-----------------------------------------------------------------------
# Maps a project into its ObsUnitSet
# @param  project    ObsProject to map
# @return Nothing
#-----------------------------------------------------------------------
#def mapprj( obsproject ):
#    action = PyProtoMapper(obsproject)  # Instantiate the action object
#    action.perform()                    # Perform the action


#-----------------------------------------------------------------------
# Stores a project into the Archive
# @param  prj   ObsProject to store
# @return the project like it got stored
#-----------------------------------------------------------------------
def storeprj (prj):
    otc = BatchOT()                    # instantiate a batch-mode OT
    action = PyStore (prj, otc)          # Instantiate the action object
    action.perform()                     # Perform the action
    prj2 = action.getProjectAfterStore() # Read the modified project
    return prj2

#-----------------------------------------------------------------------
# Stores a project into the Archive, forcing new uids
# @param  prj   ObsProject to store
# @return the project like it got stored
#-----------------------------------------------------------------------
def storenewprj (prj):
    otc = BatchOT()                    # instantiate a batch-mode OT
    asNew=1
    action = PyStore (prj, otc, asNew)   # Instantiate the action object
    action.perform()                     # Perform the action
    prj2 = action.getProjectAfterStore() # Read the modified project
    return prj2

#-----------------------------------------------------------------------
# Creates a new project
# @param  name   Name of the project
# @return A new project
#-----------------------------------------------------------------------
def newprj( name ):
    action = PyNewProject( name )       # Instanciate the action object
    action.perform()                    # Perform the action
    project = action.getObsProject()    # Retrieve the new ObsProject...
    return project                      # ...and return it


#-----------------------------------------------------------------------
# Creates a new Business Object
# 
# @param  boType     Type of the new Business Object: must be one of
#                    the constants defined in this module:
#                    TARGETSPACE, CIRCLE, RECTANGLE, ELLIPSE, POLYGON,
#                    SPECTRALELEMENT, SCHEDBLOCK or OBSUNITSET.
# @param  boParent   Business Object to which we connect the new
#                    Business Object.
# @return The new Business Object
#-----------------------------------------------------------------------
def add( boType, boParent ):
    addAction = PyAdd()
    addAction.perform( boType, boParent )   # Perform the action
    bo = addAction.getNewBO()               # Retrieve the new BO...
    # addAction returns a RegionOfInterest, not a TargetArea
    if     boType == CIRCLE                 \
        or boType == RECTANGLE              \
        or boType == ELLIPSE                \
        or boType == POLYGON:
                bo = bo.getTargetArea()
    return bo                               # ...and return it

